// Mailbox Script by Warin Cascabel - v0.5 (2 March 2009)
//
// Accepts notecards (and, using Ctrl-drag, other inventory types except for scripts). Notifies the owner via IM that a
// new message has arrived.  Because llRequestAgentData( key, DATA_ONLINE ) throws an error, and because offline IMs are
// not forwarded to email yet, this script will periodically inform the owner if there are any messages present.
//

// User-configurable variables:
//
float SecondsBetweenInforms = 900.0; // If there are messages waiting, tell the owner about them every 15 minutes.

// Internal variables
//
integer TotalObjects;
integer listenChannel;
integer listenHandle;

// Notify() tells the owner how many messages are waiting. There will always be one object in the mailbox (this script),
// so we'll ignore it.
//
Notify()
{
    TotalObjects = llGetInventoryNumber( INVENTORY_ALL );
    if (TotalObjects == 1) return;
    llInstantMessage( llGetOwner(), "You have " + (string) (TotalObjects-1) + " messages in your mailbox." );
}

default
{
    // Setup: Permit people to drop objects into the mailbox, set a timer for mail notifications, and immediately inform
    // the owner of any messages waiting.
    //
    state_entry()
    {
        llAllowInventoryDrop( TRUE );
        llSetTimerEvent( SecondsBetweenInforms );
        Notify();
    }
    
    // The timer() event handler is used for two things: periodically notifying the user of waiting messages, and timing out
    // the dialogs if no response is received within a minute. Since there is only one timer, new message notification
    // does not pick up where it left off when the dialog box was created; it waits the full interval after the dialog box
    // receives a response or times out.
    //
    timer()
    {
        if (listenHandle != 0)                        // then a dialog box is active, and has timed out.
        {
            llListenRemove( listenHandle );           // delete the listen
            listenHandle = listenChannel = 0;         // clear out the listen variables
            llOwnerSay( "Dialog timed out." );        // inform the owner of a timeout
            llSetTimerEvent( SecondsBetweenInforms ); // restart the message notification cycle
        }
        else
        {
            Notify();                                 // tell the user of any waiting messages
        }
    }
    
    // The changed() event is how we respond to inventory drops.
    //
    changed( integer what )
    {
        if (what & (CHANGED_ALLOWED_DROP | CHANGED_INVENTORY)) {       // our inventory has changed
            if (llGetInventoryNumber( INVENTORY_ALL ) > TotalObjects)  // if we have more than we used to...
            {                                                          // thank the person who dropped it (we can't
                llWhisper( 0, "Inventory accepted, thank you." );      // tell who it was, so just whisper the thanks)
                Notify();                                              // and inform the owner.
                llSetTimerEvent( 0.0 );                                // reset the timer event and start the inform
                llSetTimerEvent( SecondsBetweenInforms);               // cycle over again from this point, to avoid
            }                                                          // spamminess.
        }
    }
    
    // The touch_start() event lets the owner list, retrieve or delete messages. If anyone else touches it, he or she
    // receives a brief message indicating how to drop a message into the mailbox.
    //
    touch_start( integer foo )
    {
        key k = llDetectedKey( 0 ); // Who's touching me?!
        if (k == llGetOwner())      // Oh, it's my owner. Okay then.
        {
            if (TotalObjects == 1)  // If there are no messages, just say so and exit; no point putting up a dialog.
            {
                llOwnerSay( "You have no messages." );
                return;
            }
            if (listenHandle == 0) // Create a new listen for the dialog, unless one already exists.
            {
                listenChannel = llFloor( llFrand( 2147480000.0 )) + 3000; // Listen for responses on a random channel
                listenHandle = llListen( listenChannel, "", llGetOwner(), "" );
            }
            string prompt = "You have a total of " + (string) (TotalObjects - 1) + " new messages.";
            llSetTimerEvent( 0.0 );
            llSetTimerEvent( 60.0 ); // Time the dialog out after 60 seconds
            llDialog( llGetOwner(), prompt, [ "Retrieve", "Delete", "Cancel", "List" ], listenChannel );
        }
        else // It was someone other than my owner who touched me - give them some information.
        {
            llInstantMessage( k, "I am a mailbox! Drag a notecard onto me and I'll keep it safe for my creator." );
        }
    }

    // The listen() event handler responds to the owner's dialog choices.
    //
    listen( integer channel, string fromWhom, key id, string message )
    {
        if (id != llGetOwner()) return;     // shouldn't be necessary, but just to be safe...
        string myName = llGetScriptName();  // for ignoring this script when counting/retrieving/deleting messages
        
        // Here's an interesting OpenSim bug: constant values used in initializing a list are treated as floats.
        // Typecasting in a list initializer doesn't work in global variables in SL. It does work in OpenSim, but best
        // to make scripts as portable as possible.
        list InventoryValues = [ (integer) INVENTORY_TEXTURE, (integer) INVENTORY_SOUND, (integer) INVENTORY_LANDMARK,
                                 (integer) INVENTORY_CLOTHING, (integer) INVENTORY_OBJECT, (integer) INVENTORY_NOTECARD,
                                 (integer) INVENTORY_SCRIPT, (integer) INVENTORY_BODYPART, (integer) INVENTORY_ANIMATION,
                                 (integer) INVENTORY_GESTURE ];
        list InventoryTypes = [ "Texture", "Sound", "Landmark", "Clothing", "Object", "Notecard", "Script", "Body part",
                                "Animation", "Gesture", "Unknown Type" ];
        llListenRemove( listenHandle );                         // Delete the listen.
        listenHandle = listenChannel = 0;                       // Clear out the listen variables.
        llSetTimerEvent( 0.0 );                                 // Reset the notification cycle.
        llSetTimerEvent( SecondsBetweenInforms );
        integer objects = llGetInventoryNumber( INVENTORY_ALL); // Get the total number of objects in inventory
        if (message == "Cancel")                                // If the owner clicked the "Cancel" button...
        {
            llOwnerSay( "Canceled." );                          // ...let them know you heard them.
        }
        else if (message == "List") {                           // If the owner clicked the "List" button:
            if (objects == 1)                                   // If there's only one thing in inventory, it's this script
            {
                llOwnerSay( "No messages." );                   // so tell the owner there's no messages, and exit
                return;
            }
            integer i;                                          // Otherwise, build a list of object names, and their
            string s = "Contents:";                             // inventory types, and send the list to the owner.
            for (i = 0; i < objects; ++i)
            {
                string name = llGetInventoryName( INVENTORY_ALL, i );
                if (name != myName)
                {
                    integer t = llListFindList( InventoryValues, [ llGetInventoryType( name ) ] );
                    s += "\n" + name + " (" + llList2String( InventoryTypes, t ) + ")";
                }
                    
            }
            llOwnerSay( s );
        }
        else if (message == "Delete" )                          // If the owner clicked the "Delete" button, from either
        {                                                       // dialog...
            if (objects == 1)                                   // If there's only one thing in inventory, it's this script.
            {
                llOwnerSay( "No messages to delete." );         // so tell the owner, and exit.
                return;
            }
            while (objects--)                                                  // Delete from the end going backwards,
            {                                                                  // Otherwise it gets confused.
                string theName = llGetInventoryName( INVENTORY_ALL, objects );
                if (theName != myName)                                         // Do not delete myself!
                {
                    llOwnerSay( "Deleting " + theName );                       // Tell the owner what's being deleted
                    llRemoveInventory( theName );                              // and get rid of it.
                }
            }
            TotalObjects = 1;                                                  // We'll assume all the deletes were successful.
            llOwnerSay( "All contents deleted." );                             // Let the owner know we're finished.
        }
        else if (message == "Retrieve")                          // If the owner clicked the "Retrieve" button
        {
            integer i = objects;
            if (i == 1)                                          // Really we shouldn't need to keep checking this, but it's
            {                                                    // possible the owner may have manually deleted objects between
                llOwnerSay( "No messages to retrieve." );        // touching the mailbox and responding to the dialog.
                return;
            }
            string foldername = "Messages-" + llGetDate() + "-" + llGetSubString( llGetTimestamp(), 11, 18); // build folder name
            llOwnerSay( "Retrieving " + (string) (i-1) + " message(s) into folder " + foldername ); // print status message
            list l = [];
            while (i--)                                          // build a list of contents, except this script
            {
                string theName = llGetInventoryName( INVENTORY_ALL, i );
                if (theName != myName) l = [ theName ] + l;
            }
            llGiveInventoryList( llGetOwner(), foldername, l );             // Give the objects to the owner
            listenChannel = llFloor( llFrand( 2147480000.0 )) + 3000;       // Again, use a random channel for the dialog
            listenHandle = llListen( listenChannel, "", llGetOwner(), "" ); // Ask the user what to do with the messages
            llDialog( llGetOwner(), "Would you like to save or delete the messages?", [ "Save", "Delete" ], listenChannel );
            llSetTimerEvent( 0.0 );
            llSetTimerEvent( 60.0 );                                        // Give them a minute to answer.
        }
        else if (message == "Save") // If the user clicked the "Save" button on the secondary dialog, just send them a message.
        {
            llOwnerSay( "Messages saved. Please note that if any new messages arrive, the saved messages will be retrieved again along with the new ones." );
        }
    }
}