// PARAMOUR POLEMASTER LITE (OSSL)
// by Aine Caoimhe (c. LACM) February 2016
// Provided under Creative Commons Attribution-Non-Commercial-ShareAlike 4.0 International license.
// Please be sure you read and adhere to the terms of this license: https://creativecommons.org/licenses/by-nc-sa/4.0/
//
// This script is for the low load "Lite" version of the Polemaster
//
// *** REQUIRES REGION WITH OSSL FUNCTIONS ENABLED FOR SCRIPT OWNER
// This script requires the use of Opensim-specific OSSL functions and is not compatible with Second Life.
// This script expects each animation to have a description that identifies the sit target position (but not rotation) and time in the format <pos>time    example:  <0.00,0.00,-1.05>20.5
// Any animation not containing that information will be given default values of ZERO_VECTOR and 30.0 seconds
//
// * * * * * * * * * * *
// *   USER SETTINGS   *
// * * * * * * * * * * *
// ANIMATION SETTINGS
// These three settings affect animation handling
integer maxUsers=4;                     // how many simultaneous users are allowed (including NPCs) - more than 6 gets pretty crowded
vector globalOffset=<0.0, 0.0, -1.0>;   // position offset for the object used globally for all animations (in addition to their individual offsets)
//                                      ...Opensim 0.9 will require a value about <0,0,-0.15> lower than the identical unit in Opensim 0.8.x (and visa versa)
integer animRandomOrder=FALSE;          // TRUE = randomly select the next animation FALSE = (recommended) animations are played sequentially in the order they appear in inventory
// APPEARANCE SETTINGS
// Set the appearance values you would like to use for the dancepole
integer lightsWhenIdle=FALSE;   // TRUE = when the dance pole isn't in use, the lighting system will still continue to "play" if it's enabled -- this will put a (small) load on the sim at all times since it has to track and send those changes
                                // FALSE = lighting is frozen when the pole isn't in use (saves a small amount of sim resources)
integer mainPoleIsNeon=FALSE;   // TRUE = the main dance pole will be a neon tube that matches the floor lights in colour
                                // FALSE = it will use the texture on the base mounting bracket instead and remain that way
integer showBase=TRUE;          // TRUE = the entire base section is shown
                                // FALSE = the larger base section and 8 pot lights are hidden and only the bracket at the base of the pole is shown
                                // this overrides the showPotLights setting below
integer showPotLights=TRUE;     // TRUE will have it show the set of 8 small pot lights on the base (and chaange colour if doLighting is also TRUE above)
                                // FALSE will hide them
                                // Automatically set to FALSE if showBase=FALSE
integer showAreaLight=FALSE;    // the dance pole can also use a point lighting to somewhat simulate a spotlight shining on the general area
                                // though it eminates from the center of the pole so it's not quite as realistic as using a real spotlight and changing its colour generates a bit of additional sim load
                                // are light will be automatically disabled if there isn't one souce of light (neon pole or potlights)
integer sendLinkMessage=FALSE;  // set to TRUE if using the optional spotlights (take the spotlight from inventory and rez 1 more copies of it nearby)
                                // TRUE will send a link message on every lighting change as follows: llMessageLinked(LINK_SET,1,(vector)colour,(float)glow)
                                // FALSE disables this
integer randomColour=TRUE;      // TRUE will randomly select a colour each time the colour changes...FALSE will change colour in the sequence of the colour list
float lightTimer=3.0;           // How often (in seconds) to change the colour of any lights....shorter times place a little more load on the sim due to prim properties..somewhere in the 3 to 5 second range looks decent
                                // avoid values under 2 seconds...only works if doLighting is TRUE above
list colours=[                  // a list of colours and matching glow settings to use for the pot lights and/or pole -- supply as many as you like (requires at least 3)
                <1.00, 0.00, 0.00>, 0.25,
                <1.00, 0.50, 0.00>, 0.25,
                <1.00, 1.00, 0.00>, 0.25,
                <0.50, 1.00, 0.00>, 0.275,
                <0.00, 1.00, 0.00>, 0.30,
                <0.00, 1.00, 0.50>, 0.35,
                <0.00, 1.00, 1.00>, 0.40,
                <0.00, 0.50, 1.00>, 0.55,
                <0.00,0.00, 1.00>, 0.75,
                <0.50, 0.00, 1.00>, 0.50,
                <1.00, 0.00, 1.00>, 0.35,
                <1.00, 0.00, 0.50>, 0.30,
                <1.00, 1.00, 1.00>, 0.25
            ];
float neonAlpha=0.8;            // alpha of the pot lights if visible, and of the pole if it set to neon
float poleGlowMultiplier=0.5;   // the colour's glow value is multiplied by this value when used on the main pole
// IF USING showAreaLight = TRUE the next 3 settings affect the area light settings (just like setting them using the "Features" tab of your viewer's editor on a prim)
float intesityMultiplier=3.0;   // if using showAreaLight=TRUE, the glow setting for the current light colour is multiplied by this value to determine the light intensity of the area light
                                // this multiplier is applied globally for all colours, with a maximum resulting intensity value of 1.0...usually somewhere in the 1.25 to 2.0 range looks good
float lightRadius=2.5;          // if using showAreaLight=TRUE, the radius of the area light (avoid using large values so you don't wash out bathe other nearby objects in unwanted light)
float lightFalloff=0.75;        // if using showAreaLight-TRUE, the falloff value of the area light (values around 0.75 generally look best)
//
// # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
// # #    MAIN SCRIPT STARTS HERE - DO NOT CHANGE ANYTHING BELOW THIS LINE UNLESS YOU KNOW WHAT YOU'RE DOING!    # #
// # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
integer debug=FALSE;    // TRUE displays animation info as floating text above the pole during use
float animTimer;
vector sitOffset;
list userIDs;
list animList;
integer animInd;
integer faceBaseMain=1;
integer faceBaseCenter=0;
integer faceBaseMount=4;
integer faceBasePole=3;
integer faceBasePotLights=2;
integer currentColour;
integer doLighting;

playNextAnim()
{
    llSetTimerEvent(0.0);
    string anToStop=llList2String(animList,animInd);
    animInd+=3;
    if (animInd>=llGetListLength(animList))
    {
        if (animRandomOrder)
        {
            animList=[]+llListRandomize(animList,3);
            while ((llList2String(animList,0)==anToStop) && (llGetListLength(animList)>1)) { animList=[]+llListRandomize(animList,3); }
        }
        animInd=0;
    }
    integer u=llGetListLength(userIDs);
    float curU=(float)u;
    string anToStart=llList2String(animList,animInd);
    sitOffset=llList2Vector(animList,animInd+1)+globalOffset;
    while(--u>=0)
    {
        key who=llList2Key(userIDs,u);
        osAvatarPlayAnimation(who,anToStart);
        osAvatarStopAnimation(who,anToStop);
        positionUpdate((360.0/curU)*u,who);
    }
    llSetTimerEvent(llList2Float(animList,animInd+2));
    if (debug)
    {
        string txt=llDumpList2String(llList2List(animList,animInd,animInd+2)," | ");
        llSetText(txt,<1.0,0.0,0.0>,1.0);
        llOwnerSay(txt);
    }
}
startAnimations()
{
    integer u=llGetListLength(userIDs);
    float curU=(float)u;
    string anToStart=llList2String(animList,animInd);
    sitOffset=llList2Vector(animList,animInd+1)+globalOffset;
    while (--u>-1)
    {
        key who=llList2Key(userIDs,u);
        positionUpdate((360.0/curU)*u,who);
        list animToStop=llGetAnimationList(who);
        integer an=llGetListLength(animToStop);
        osAvatarPlayAnimation(who,anToStart);
        while(--an>-1) { osAvatarStopAnimation(who,llList2Key(animToStop,an)); }
    }
    playNextAnim();
}
stopAnimations()
{
    list sitters;
    integer link=llGetNumberOfPrims();
    while(link>1)
    {
        if (llGetAgentSize(llGetLinkKey(link))!=ZERO_VECTOR) sitters=[]+sitters+[llGetLinkKey(link)];
        else link=1;
        link--;
    }
    integer u=llGetListLength(userIDs);
    while (--u>-1)
    {
        if (llListFindList(sitters,[llList2Key(userIDs,u)])==-1) // no longer using
        {
            osAvatarPlayAnimation(llList2Key(userIDs,u),"Stand");
            osAvatarStopAnimation(llList2Key(userIDs,u),llList2String(animList,animInd));
            userIDs=[]+llDeleteSubList(userIDs,u,u);
        }
    }
    if (llGetListLength(userIDs)<1)
    {
        if (debug) llSetText("Ready",<1,0,0>,1.0);
    }
}
positionUpdate(float zRotation,key who)
{
    vector pos=sitOffset;
    vector size = llGetAgentSize(who);
    integer linkNum = llGetNumberOfPrims();
    do
    {
        if(who == llGetLinkKey( linkNum ))
        {
            rotation rot=llEuler2Rot(<0,0,zRotation>*DEG_TO_RAD);
            float fAdjust = ((((0.008906 * size.z) + -0.049831) * size.z) + 0.088967) * size.z;
            vector newPos=((pos + <0.0, 0.0, 0.4>) - (<0.0,0.0,1.0> * fAdjust))*rot;
            llSetLinkPrimitiveParamsFast(linkNum, [PRIM_POS_LOCAL,newPos,PRIM_ROT_LOCAL, rot]);
            jump end;
        }
    }while( --linkNum );
    @end;
}
doBuildAnimationList()
{
    animList=[];
    integer i=llGetInventoryNumber(INVENTORY_ANIMATION);
    while (--i>-1)
    {
        string name=llGetInventoryName(INVENTORY_ANIMATION,i);
        string desc=osGetInventoryDesc(name);
        integer ev=llSubStringIndex(desc,">");
        vector off=ZERO_VECTOR;
        float time=30.0;
        if (ev>=6)  // expected
        {
            off=(vector)llGetSubString(desc,0,ev);
            time=(float)llGetSubString(desc,ev+1,-1);
            if (time<=0) time=30.0;
        }
        animList=[]+animList+[name,off,time];
    }
    if (animRandomOrder) animList=[]+llListRandomize(animList,3);
    else animList=[]+llListSort(animList,3,TRUE);
    animInd=llGetListLength(animList);
    if (debug) llOwnerSay("Found "+(string)(animInd/3)+" animations");
    animInd-=3;
    if (llGetListLength(animList)<3) llOwnerSay("ERROR! Did not find any animations in inventory! Please add some and reset the script");
}
setNextColour()
{
    if (randomColour) currentColour=2*llFloor(llFrand((float)llGetListLength(colours)) / 2.0);
    else
    {
        currentColour+=2;
        if (currentColour>=llGetListLength(colours)) currentColour=0;
    }
    vector colourToSet=llList2Vector(colours,currentColour);
    float glow=llList2Float(colours,currentColour+1);
    if (sendLinkMessage) llMessageLinked(LINK_SET,1,(string)colourToSet,(string)glow);
    if (mainPoleIsNeon) llSetLinkPrimitiveParamsFast(LINK_THIS,[PRIM_COLOR,faceBasePole,colourToSet,neonAlpha,PRIM_GLOW,faceBasePole,glow*poleGlowMultiplier]);
    if (showPotLights && showBase) llSetLinkPrimitiveParamsFast(LINK_THIS,[PRIM_COLOR,faceBasePotLights,colourToSet,neonAlpha,PRIM_GLOW,faceBasePotLights,glow]);
    if (showAreaLight && (mainPoleIsNeon || showPotLights)) llSetLinkPrimitiveParamsFast(LINK_THIS,[PRIM_POINT_LIGHT,showAreaLight,llList2Vector(colours,currentColour),intesityMultiplier*llList2Float(colours,currentColour+1),lightRadius,lightFalloff]);
}
activateLighting()
{
    integer alreadyActive=doLighting;
    integer links=llGetNumberOfPrims();
    if (links==1)
    {
        if (lightsWhenIdle) doLighting=TRUE;
        else doLighting=FALSE;
    }
    else if (llGetAgentSize(llGetLinkKey(links))!=ZERO_VECTOR) doLighting=TRUE;
    else if (lightsWhenIdle) doLighting=TRUE;
    else doLighting=FALSE;
    if (doLighting && !alreadyActive)
    {
        setNextColour();
        llSensorRepeat("NO_RESULT_POSSIBLE",NULL_KEY,0,0.01,0.01,lightTimer);
    }
    else if (alreadyActive && !doLighting) llSensorRemove();
}
matchPoleToBracket()
{
    list dataToSet=llGetLinkPrimitiveParams(LINK_THIS,[PRIM_TEXTURE,faceBaseMount,PRIM_COLOR,faceBaseMount,PRIM_BUMP_SHINY,faceBaseMount,PRIM_FULLBRIGHT,faceBaseMount,PRIM_GLOW,faceBaseMount]);
    llSetLinkPrimitiveParamsFast(LINK_THIS,[PRIM_TEXTURE,faceBasePole]+llList2List(dataToSet,0,3)+[PRIM_COLOR,faceBasePole]+llList2List(dataToSet,4,5)+[PRIM_BUMP_SHINY,faceBasePole]+llList2List(dataToSet,6,7)+[PRIM_FULLBRIGHT,faceBasePole,llList2Integer(dataToSet,8),PRIM_GLOW,faceBasePole,llList2Float(dataToSet,9)]);
}
setNeonPole()
{
   llSetLinkPrimitiveParamsFast(LINK_THIS,[PRIM_TEXTURE,faceBasePole]+llGetLinkPrimitiveParams(LINK_THIS,[PRIM_TEXTURE,faceBasePotLights])+[PRIM_COLOR,faceBasePole,llList2Vector(colours,currentColour),neonAlpha,PRIM_BUMP_SHINY,faceBasePole,PRIM_SHINY_NONE,PRIM_BUMP_NONE,PRIM_FULLBRIGHT,faceBasePole,TRUE,PRIM_GLOW,faceBasePole,poleGlowMultiplier*llList2Float(colours,currentColour+1)]);
}
doShowHideBase()
{
    llSetLinkAlpha(LINK_THIS,(float)showBase,faceBaseMain);
    llSetLinkAlpha(LINK_THIS,(float)showBase,faceBaseCenter);
    llSetLinkPrimitiveParams(LINK_THIS,[PRIM_COLOR,faceBasePotLights,llList2Vector(colours,currentColour),(float)showBase*(float)doLighting*neonAlpha,PRIM_GLOW,faceBasePotLights,(float)showBase*llList2Float(colours,currentColour+1)]);
}

default
{
    state_entry()
    {
        if (maxUsers==0)
        {
            llOwnerSay("ERROR! Maximum users is set at 0 which doesn't allow anyone to use this. Setting it to 1 instead.");
            maxUsers=1;
        }
        userIDs=[];
        doBuildAnimationList();
        if (sitOffset==ZERO_VECTOR) sitOffset=<0,0,0.000001>;
        llSitTarget(sitOffset,ZERO_ROTATION);
        llSetClickAction(CLICK_ACTION_SIT);
        if (debug) llSetText("Ready",<1.0,0.0,0.0>,1.0);
        else llSetText("",<0,0,0>,0.0);
        if (mainPoleIsNeon) setNeonPole();
        else matchPoleToBracket();
        if (llGetNumberOfPrims()>1)
        {
            if (llGetLinkName(2)=="cage (optional)")
            {
                llSetLinkAlpha(2,(float)showBase,ALL_SIDES);
                if (!showBase) llOwnerSay("If you expect to keep this dancepole with the cage permanently hidden I recommend that you unlink and then delete the cage to reduce the data that needs to be sent to your guests");
            }
        }
        doShowHideBase();
        // force-set area light in case it was changed from on to off
        llSetLinkPrimitiveParamsFast(LINK_THIS,[PRIM_POINT_LIGHT,showAreaLight,llList2Vector(colours,currentColour),intesityMultiplier*llList2Float(colours,currentColour+1),lightRadius,lightFalloff]);
        activateLighting();
        setNextColour();
    }
    no_sensor() // expected result
    {
        setNextColour();
    }
    sensor(integer num)
    {
        llOwnerSay("Seriously?!"); // should be impossible but if it somehow does, remove sensor
        llSensorRemove();
    }
    timer()
    {
        playNextAnim();
    }
    on_rez(integer start_param)
    {
        llResetScript();
    }
    changed(integer change)
    {
        if (change & CHANGED_OWNER) llResetScript();
        if (change & CHANGED_REGION_START) llResetScript();
        if (change & CHANGED_LINK)
        {
            if(llGetAttached()>0) return;
            activateLighting();
            list sitters;
            integer link=llGetNumberOfPrims();
            while(link)
            {
                if (llGetAgentSize(llGetLinkKey(link))!=ZERO_VECTOR) sitters+=[llGetLinkKey(link)];
                else link=1;
                link--;
            }
            if (llGetListLength(sitters)==0) stopAnimations();
            else
            {
                if (llGetListLength(sitters)>maxUsers)
                {
                    integer s=llGetListLength(sitters);
                    while (--s>-1)
                    {
                        if (llListFindList(userIDs,[llList2Key(sitters,s)])==-1)
                        {
                            llRegionSayTo(llList2Key(sitters,s),0,"Sorry, the pole is already at its maximum allowed number of users");
                            llUnSit(llList2Key(sitters,s));
                            sitters=[]+llDeleteSubList(sitters,s,s);
                        }
                    }
                }
                else
                {
                    stopAnimations();
                    userIDs=[]+sitters;
                    startAnimations();
                }
            }
        }
    }
}