// PARAMOUR REMOTE SPOTLIGHT CONTROLLER
// by Aine Caoimhe (LACM) February 2016)
// Provided under Creative Commons Attribution-Non-Commercial-ShareAlike 4.0 International license.
// Please be sure you read and adhere to the terms of this license: https://creativecommons.org/licenses/by-nc-sa/4.0/
//
// This script is intended as an option remote spolight controller for use with the Paramour Polemaster Lite
//
// INSTRUCTIONS:
// 1. Ensure that OSSL is enabled in the region and that you have permission to use the following functions to allow it to set the prim properties:
//          - osSetPrimitiveParamas()
//          - osSetProjectionParams()
// 2. Place this script into the Paramour Polemaster Lite dance pole (that contains the main script and all the aniamtions)
// 3. Rez one or more Paramour Remote Spoilights nearby and ppoint them at the polemaster. Make sure they're within the range set in the user options settings below.
// 4. Reset this master script
// 5. The next time the main dancepole's lights change, the spotlight will change to match
//
// USER OPTIONS
integer audible=FALSE;  // TRUE = will tell you in chat when reset whether it was able to find any nearby spotlights...FALSE will not
string objectName="Paramour Remote Spotlight";  // object name of the spotlight to look for. By default it's "Paramour Remote Spotlight" but if you rename the spotlights you'll need to change this to match your change
string spotlightTexture="spotlight v2"; // name of the texture to use for the spotlight projection (default is the one I supply)...must be in the prim's inventory
float range = 15.0; // maximum distance from the Polemaster Lite to look for spotlights...keep this range relatively small...in most cases 15m ought to be fine since spotlight light range is only about 10m max...
//                     In the default Opensim set-up you can only detect up to 16 spotlights and in practice any more than 4 is pointless
// The values below will depend a little on what your region windlight settings are, and how many spotlights you're using
// These are the settings that will be applied to each spotlight (exactly as though doing it via the "Features" tab in prim edit)
// You can edit and change them during use of the dancepole and they will be updated on all spotlights when the next colour change of the dancepole occurs
float intensity=0.6;
float radius=5.0;
float falloff=0.75;
float fov=1.5;
float focus=5.0;
float ambiance=0.05;
//
//
// * * * * * * * * * * * ** * * * * * * * * * * * * * * * * * * * * * * * *
// DON'T CHANGE ANYTHING BELOW THIS LINE UNLESS YOU KNOW WHAT YOU'RE DOING!
// * * * * * * * * * * * ** * * * * * * * * * * * * * * * * * * * * * * * *
integer lightFace=0; // face of the bulb in the spotlight
list spotlights;
key spotKey;

findSpotlights()
{
    spotKey=llGetInventoryKey(spotlightTexture);
    spotlights=[];
    llSensor(objectName,NULL_KEY,PASSIVE,range,PI);
}
default
{
    state_entry()
    {
        findSpotlights();
    }
    on_rez(integer foo)
    {
        llResetScript();
    }
    changed (integer change)
    {
        if (change & CHANGED_REGION_START) llResetScript();
        else if (change & CHANGED_OWNER) llResetScript();
    }
    no_sensor()
    {
        if (audible) llOwnerSay("WARNING! Unable to detect any spotlights in range");
    }
    sensor(integer num)
    {
        while (--num>=0) {spotlights=[]+spotlights+[llDetectedKey(num)];}
        if (audible) llOwnerSay("Spotlights detected: "+(string)llGetListLength(spotlights));
    }
    link_message (integer link, integer num, string colour, key glow)
    {
        // expected message from the main script on each lighting change: llMessageLinked(LINK_SET,1,(string)colourToSet,(string)glow);
        if (num!=1) return;
        integer i=llGetListLength(spotlights);
        while (--i>=0)
        {
            key target=llList2Key(spotlights,i);
            if (llList2String(llGetObjectDetails(target,[OBJECT_NAME]),0)!=objectName)
            {
                if (audible) llOwnerSay("Spotlight "+(string)target+" seems to have been removed from ther scene");
                spotlights=[]+llDeleteSubList(spotlights,i,i);
            }
            else
            {
                osSetPrimitiveParams(target,[PRIM_GLOW,lightFace,(float)glow,PRIM_COLOR,lightFace,(vector)colour,1.0,PRIM_POINT_LIGHT,TRUE,(vector)colour,intensity,radius,falloff]);
                osSetProjectionParams(target,TRUE,spotKey,fov,focus,ambiance);
            }
        }
    }
}