/** Simple chess board  script by Satyr Aeon  
 **
 **/
 
list board = [];  
list boardLinks = [];
list boardHasMoved = [];
list outPieces;


integer currentColor; 
integer otherColor; 

integer WHITE = 1;
integer BLACK = 2;
integer PAWN  = 4;
integer BISHOP = 8;
integer ROOK   = 16;
integer KNIGHT = 32;
integer QUEEN  = 64;
integer KING   = 128;

integer cur  = -1;
integer prev = -1;

key whitePlayer = NULL_KEY;
key blackPlayer = NULL_KEY;

key waitUserToResign = NULL_KEY;

key currentPlayer = NULL_KEY;

integer enPassantVictim;
integer enPassantTarget;
integer pendingPromotion=-1;

integer listener=-1;
integer listenTs;

integer chan(key u)
{
    return -1 - (integer)("0x" + llGetSubString( (string) u, -6, -1) )-393;
}

string colorName(integer w)
{
    if (w & WHITE) return "White";
    else return "Black";
}

string pieceName(integer w)
{
    if (w &PAWN) return "Pawn";
    else if (w &KING) return "King";
    else if (w &ROOK) return "Rook";
    else if (w &KNIGHT) return "Knight";
    else if (w &QUEEN) return "Queen";
    else if (w &BISHOP) return "Bishop";
    return "Unknown";
}

string pos2string(integer l)
{
    return llGetSubString("abcdefgh", l%8, l%8) +  (string) ((integer)(1+ l/8));
}

sound(string m)
{
    llTriggerSound(m, 1.0);  //  llTriggerSound(m, 1.0);    llTriggerSound(m, 1.0);
}
startListen()
{
    if (listener<0) 
    {
        listener = llListen(chan(llGetKey()), "", "", "");
        listenTs = llGetUnixTime();
    }
}

checkListen()
{
    if (listener > 0 && llGetUnixTime() - listenTs > 300)
    {
        llListenRemove(listener);
        listener = -1;
    }
}

integer sq(vector pos)
{
    return llList2Integer(board, (integer) (8*pos.y+pos.x)); 
}

setBoard(integer idx, integer piece)
{
    board = [] + llListReplaceList(board, [piece], idx, idx);
}

integer getBoardLink(integer idx)
{
    return llList2Integer(boardLinks, idx);
}

setBoardLink(integer idx, integer lnk)
{
    boardLinks= [] + llListReplaceList(boardLinks, [lnk], idx, idx);
}


setHasMoved(integer idx, integer val)
{
    boardHasMoved = [] + llListReplaceList(boardHasMoved, [val], idx, idx);
}

integer getHasMoved(integer idx)
{
    return llList2Integer(boardHasMoved, idx);
}


integer getLink(string name)
{
    integer i;
    for (i=2; i <= llGetNumberOfPrims(); i++) if (llGetLinkName(i) == name) return i;
    return -1;
}

vector  pos2vec(integer pos)
{
    return < pos%8, pos/8, 0>;
}

integer vec2pos(vector v)
{
    return (integer)(v.y*8 + v.x);
}

integer isInBounds(vector v)
{
    if (v.x>=0 && v.x<8 && v.y>=0 && v.y<8) return TRUE;
    return FALSE;
}


string toLinkName(integer p)
{
    string s;
    if (p &WHITE) s = "W";
    else s="B";
    if (p & KING) s+= "K";
    else if (p & QUEEN) s+= "Q";
    else if (p & BISHOP) s+= "B";
    else if (p & KNIGHT) s+= "N";
    else if (p & PAWN) s+= "P";
    else if (p & ROOK) s+= "R";
    return s;
}

initBoard()
{
    board = [
    WHITE|ROOK, WHITE|KNIGHT, WHITE|BISHOP,WHITE|QUEEN, WHITE|KING, WHITE|BISHOP, WHITE|KNIGHT, WHITE|ROOK,
    WHITE|PAWN, WHITE|PAWN, WHITE|PAWN, WHITE|PAWN, WHITE|PAWN, WHITE|PAWN, WHITE|PAWN, WHITE|PAWN, 
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    BLACK|PAWN, BLACK|PAWN, BLACK|PAWN, BLACK|PAWN, BLACK|PAWN, BLACK|PAWN, BLACK|PAWN, BLACK|PAWN, 
    BLACK|ROOK, BLACK|KNIGHT, BLACK|BISHOP,BLACK|QUEEN, BLACK|KING, BLACK|BISHOP, BLACK|KNIGHT, BLACK|ROOK
    ];
    
    outPieces = [];
    
    boardHasMoved = [0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0];
    
    boardLinks = [];
    integer i;
    list used= [];

    for (i=0; i < llGetListLength(board); i++)
    {
        
        integer p = llList2Integer(board,i);
        if (p != 0) 
        {
            string s = toLinkName(p);
            integer j;
            for ( j=2; j <= llGetNumberOfPrims(); j++)
            {
                if (llGetLinkName(j) == s)
                {
                    if (llListFindList(used, [j])==-1)
                    {
                        boardLinks += j;
                        used += j;
                        llSetLinkPrimitiveParamsFast(j, [PRIM_TEXT, "", <1,1,1>, 1.0]);
                        jump foundPiece;
                    }
                }
            }
            @foundPiece;
        }
        else
            boardLinks += 0;
    }
    
    enPassantTarget=0;
    pendingPromotion=-1;
    currentPlayer = whitePlayer;
}

integer canMove(integer pos1, integer pos2)
{
    integer p1 = llList2Integer(board, pos1);
    integer p2 = llList2Integer(board, pos2);
    vector v1 = pos2vec(pos1); 
    vector v2 = pos2vec(pos2); 

    if (p1 & PAWN)
    {
        integer dir;
        if (p1 & WHITE) 
        {
            if (v2.y <= v1.y || v1.y>=7) return FALSE; 
            dir = 1;
        }
        else
        {
            if (v2.y >= v1.y || v1.y<=0) return FALSE; 
            dir = -1;
        }

        if (p2 ==0)
        {
            if (v2.x == v1.x) {
                if ( (v2.y == v1.y+dir)) {  return TRUE;  } 

                if ( getHasMoved(pos1)==0 && sq(v1 + <0,dir,0>)==0 && (v2.y == v1.y+dir+dir) ) {
                    enPassantTarget = vec2pos( v1 + <0,dir,0> ); // Set flag for possible en passant in next move
                    enPassantVictim = pos2 ; 
                    
                    return TRUE; 
                }     
            }
            
            // Check en passant                
            if ( pos2 == enPassantTarget)
                return TRUE;

        }
        else // Capture
        {
            if ( (v2.x == v1.x-1  || v2.x == v1.x+1) && v2.y == v1.y+dir) { return TRUE; } 
        }
    }
    else if (p1 & (QUEEN|ROOK|BISHOP|KING))
    {
        if ( (  (p1 & (QUEEN|KING)) && ( (v2.y == v1.y) || v2.x == v1.x || (llFabs(v2.x-v1.x) == llFabs(v2.y - v1.y) ) ) )
         || ( (p1 & ROOK) &&  ( (v2.y == v1.y) || v2.x == v1.x ) )
         || ( (p1 & BISHOP) &&  ( llFabs(v2.x-v1.x) == llFabs(v2.y - v1.y) ) ) )
        {
            integer dirX =0;
            if (v1.x < v2.x) dirX = 1;
            else if (v1.x > v2.x) dirX = -1;

            integer dirY =0;
            if (v1.y < v2.y) dirY = 1;
            else if (v1.y > v2.y) dirY = -1;

            vector cp = v1;
            integer limit = 8;
            if (p1 & KING) limit =1;
            integer i;
            for (i=0; i < limit; i++)
            {
                cp += <dirX, dirY, 0>;
                if (isInBounds(cp))
                {
                    if (cp == v2) {  return TRUE; }
                    else if  (sq( cp) != 0) {  return FALSE; }
                }
            }
        }
    }
    else if (p1 & KNIGHT)
    {
        if ( (llFabs(v2.y-v1.y) == 2 && llFabs(v2.x-v1.x)==1)
         ||  (llFabs(v2.y-v1.y) == 1 && llFabs(v2.x-v1.x)==2))
         {
             return TRUE;
         }
    }
    
    return FALSE;
}


refreshBoard()
{
    vector sz = llGetScale();
    integer i;
    for (i=0; i < llGetListLength(board); i++)
    {
        integer p = llList2Integer(boardLinks,i);
        if (p != 0)
        {
            vector xy = pos2vec(i)/8.;
            vector curPos = llList2Vector(llGetLinkPrimitiveParams(p, [PRIM_POS_LOCAL]),0);
            vector pos = < sz.x*xy.x - sz.x/2 + sz.x/16, sz.y*xy.y - sz.y/2 + sz.y/16, curPos.z>;
            llSetLinkPrimitiveParamsFast(p, [PRIM_POS_LOCAL, pos]);
            
            if (llGetLinkName(p) != toLinkName(llList2Integer(board, i)))
            {
                // This must be a promoted piece
                llSetLinkPrimitiveParamsFast(p, [PRIM_TEXT, pieceName(llList2Integer(board,i)), <1,1,.1>, 1.0 ]); 
            }
        }
    }
    
    for (i=0; i < llGetListLength(outPieces); i++)
    {
            vector curPos = llList2Vector(llGetLinkPrimitiveParams(llList2Integer(outPieces,i), [PRIM_POS_LOCAL]),0);
            vector pos = < sz.x/2 +1 +i*.5, 0, curPos.z>;
            llSetLinkPrimitiveParamsFast( llList2Integer(outPieces,i) , [PRIM_POS_LOCAL, pos]);
    }


    llSetLinkPrimitiveParamsFast( getLink("white_control"),  [PRIM_GLOW, ALL_SIDES, (currentColor ==WHITE)*.5]);
    llSetLinkPrimitiveParamsFast( getLink("black_control"),  [PRIM_GLOW, ALL_SIDES, (currentColor ==BLACK)*.5]);
        
    string s;
    if (whitePlayer != NULL_KEY && blackPlayer != NULL_KEY)
        s += llKey2Name(whitePlayer)+ " vs "+llKey2Name(blackPlayer);
    else 
    {
     if (whitePlayer!= NULL_KEY) s+= llKey2Name(whitePlayer)+" plays white\nClick to join.";
        else if (blackPlayer!= NULL_KEY) s+= llKey2Name(blackPlayer)+" plays black\nClick to join.";
        else s += "Click to join";
    }
    setText(s, <.5, 1,.5>);
}

setText(string s, vector c)
{
    llSetLinkPrimitiveParamsFast( getLink("board_control"),  [PRIM_TEXT, s, c, 1.]);
}

integer prevCurrent =-1;
setCurrent(integer idx)
{
    if (prevCurrent>0)
        llSetLinkPrimitiveParamsFast(prevCurrent, [PRIM_GLOW, ALL_SIDES, 0] );        
    if (idx>=0)
    {
        integer lnk = llList2Integer(boardLinks, idx);
        llSetLinkPrimitiveParamsFast(lnk, [PRIM_GLOW, ALL_SIDES, 0.5] );
        prevCurrent = lnk;
    }
}



swapPositions(integer pos1, integer pos2)
{
    integer p1 = llList2Integer(board, pos1);
    integer p2 = llList2Integer(board, pos2);
    integer lnk1 = llList2Integer(boardLinks, pos1);
    integer lnk2 = llList2Integer(boardLinks, pos2);

    setBoard(pos1, p2);
    setBoard(pos2, p1);
    setBoardLink(pos2, lnk1);
    setBoardLink(pos1, lnk2);
    setHasMoved(pos1, 1);
    setHasMoved(pos2, 1);
}

resignUser(key userToResign)
{
    if (userToResign == whitePlayer || userToResign == blackPlayer)
    {
        key winner = whitePlayer;
        if (userToResign == whitePlayer) winner= blackPlayer;
        llSay(0, llKey2Name(whitePlayer) + " won!");
        
        setText(llKey2Name(winner) + " won " + llKey2Name(userToResign), <.5,1,.5> );
    }
    whitePlayer = NULL_KEY;
    blackPlayer = NULL_KEY;
    currentPlayer = NULL_KEY;
}


integer doCastle( string m)
{
    
    list moveFromPos;
    list moveToPos;
    list mustBeEmpty;
    
    if (currentColor == WHITE)
    {
        if (m == "Queenside")
        {
            moveFromPos = [0, 4];
            moveToPos = [3, 2];
            mustBeEmpty = [1,2,3];
        }
        else if (m == "Kingside")
        {
            moveFromPos = [7, 4];
            moveToPos = [5, 6];
            mustBeEmpty = [5,6];
        }
    }
    else if (currentColor == BLACK)
    {
        if (m == "Queenside")
        {
            moveFromPos = [56, 60];
            moveToPos = [59, 58];
            mustBeEmpty = [57,58,59];
        }
        else if (m == "Kingside")
        {
            moveFromPos = [63, 60];
            moveToPos = [61, 62];
            mustBeEmpty = [61,62];
        }
    }
    
    integer i;
    for (i=0;i < llGetListLength(mustBeEmpty); i++)
        if (llList2Integer(board, llList2Integer(mustBeEmpty,i)) != 0)
        {
            llSay(0,  pos2string(llList2Integer(mustBeEmpty,i))+" is not empty");
            return FALSE;
        }
    for (i=0;i < llGetListLength(moveFromPos); i++)
        if (getHasMoved(llList2Integer(moveFromPos, i)))
        {
            llSay(0, "Piece at "+pos2string(llList2Integer(moveFromPos,i)) + " has moved! Cannot castle");
            return FALSE;
        }


    swapPositions( llList2Integer(moveFromPos,0), llList2Integer(moveToPos,0) );
    swapPositions( llList2Integer(moveFromPos,1), llList2Integer(moveToPos,1) );

    return TRUE;
}

switchSides()
{
    integer tmp = currentColor;
    currentColor = otherColor;
    otherColor = tmp;
    
    if (currentColor == WHITE) currentPlayer = whitePlayer;
    else currentPlayer = blackPlayer;
    setCurrent(-1);
}

promotionDialog(key u)
{
    startListen();
    llDialog(u, "Select promotion", ["Bishop", "Rook", "Queen", "Knight"], chan(llGetKey()));
    llSetTimerEvent(300);
}

default 
{
    state_entry()
    {
        initBoard();
        currentColor = 0;
        otherColor = 0;
        refreshBoard();
        
    }

    touch_start(integer n)
    {
        string nm = llGetLinkName(llDetectedLinkNumber(0));
        key u = llDetectedKey(0);
        
        
        if (nm =="board_control")
        {
            if (llGetAgentSize(whitePlayer)==ZERO_VECTOR)  whitePlayer = NULL_KEY;
            if (llGetAgentSize(blackPlayer)==ZERO_VECTOR)  blackPlayer = NULL_KEY;

            list opts = ["CLOSE", "Help", "."];
            if (whitePlayer == NULL_KEY || blackPlayer ==NULL_KEY)
            {
                opts += "Join";
                opts += "Quit";
            }
            else
            {
                if (u == currentPlayer)
                {
                    opts += "Resign";
                    opts += "Castle...";
                }
                else
                    opts += "Checkmate";
            }
            
            startListen();    
            llDialog(llDetectedKey(0), "Menu", opts, chan(llGetKey()));
            llSetTimerEvent(300);
        }
        else if ( currentPlayer == u) 
        {

            integer nidx = -1;
            if (llDetectedLinkNumber(0) == 1) // board
            {
                vector tf =llDetectedTouchST(0);
                integer cX = (integer)(tf.x*8.);
                integer cY = (integer)(tf.y*8.);
                nidx = vec2pos(<cX,cY,0>);

 
            }
            else
            {
                vector sz = llGetScale();
                vector p = llList2Vector(llGetLinkPrimitiveParams(llDetectedLinkNumber(0), [PRIM_POS_LOCAL]),0);
                p += sz *.5;
                integer cX = (integer)((p.x / sz.x)*8.);
                integer cY = (integer)((p.y / sz.y)*8.);
                nidx = vec2pos(<cX,cY,0>);
                
            }           
            
            if (nidx == -1) return;
            if (pendingPromotion>=0)
            {
                promotionDialog(currentPlayer);
                return;
            }
            else if (cur==-1) 
            {
                if (llList2Integer(board, nidx) ==0) return;
                if ( llList2Integer(board, nidx) & otherColor)
                {
                    sound("doing");
                    llSay(0, "Not your color!");
                    return;
                }
                sound("click");
                cur = nidx;
                setCurrent(cur);
            }
            else if (cur == nidx)
            {
                cur = -1;
                setCurrent(-1); // deselect
            }
            else if (cur >=0)
            {
                integer p1 = llList2Integer(board, cur);
                integer p2 = llList2Integer(board, nidx);
                
                if ( (p1 & p2 & currentColor)>0)
                {
                    cur = -1;
                    return;
                }

                if ( (p1&currentColor) && ( (p2&otherColor) || p2 ==0) 
                    &&  canMove(cur, nidx) )
                {
                    // Do move
                    swapPositions(cur, nidx);
                    // check if it s a capture
                    integer lnkNew = getBoardLink(cur);
                    if (lnkNew !=0)  
                    {
                        outPieces += lnkNew;
                        setBoardLink(cur, 0);
                        setBoard(cur, 0);
                        if (p2&KING)
                        {
                            llSay(0, "King is slain!");
                            if (currentPlayer == whitePlayer)
                                resignUser(blackPlayer);
                            else
                                resignUser(whitePlayer);
                            setCurrent(-1);
                            refreshBoard();
                            sound("hit_open");
                            return;
                        }
                    }
                    
                    if ( (p1 & PAWN) && enPassantTarget == nidx)
                    {
                        integer lnk3 = getBoardLink(enPassantVictim);
                        if ( llList2Integer(board, enPassantVictim) & (PAWN | otherColor) )
                        {
                            outPieces += lnk3;
                            setBoard(enPassantVictim, 0);
                            setBoardLink(enPassantVictim, 0);
                            llSay(0, "Capturing en passant piece");
                            enPassantTarget = -1;
                        }
                        else { llSay(0, "ERROR should not happen"); }
                    }
                    if (enPassantTarget>0 && ( (llList2Integer(board, enPassantVictim )&currentColor) ==0) )
                    {
                        enPassantTarget = -1; // Reset after next move or if otherwise captured
                    }                    

                    llSay(0, colorName(currentColor)+" moves "+pieceName(p1) + " from "+ pos2string(cur)+" to "+pos2string(nidx)+". "+ colorName(otherColor)+"'s turn");
                    
                    if ( (p1 == (PAWN|WHITE) && nidx>55 ) || ( p1 == (PAWN|BLACK) && nidx<8) )
                    {
                        pendingPromotion = nidx;
                        sound("lap");
                        refreshBoard();
                        llSay(0, "Waiting for pawn promotion");
                        promotionDialog(currentPlayer);
                        refreshBoard();
                        return;
                    }

                    cur = nidx = -1;
                    
                    switchSides();
                    refreshBoard();

                    if (lnkNew !=0)
                        sound("hit_kill");
                    else
                        sound("hit_move");

                    if (waitUserToResign != NULL_KEY)
                    {
                        waitUserToResign = NULL_KEY;
                    }

                }
                else
                {
                    cur = nidx = -1;
                    setCurrent(-1);
                    sound("doing");
                    llSay(0, "Illegal!");
                    return;
                }
            }
        }
    }


    listen(integer c, string nm, key id, string m)
    {
        if (m == "CLOSE")
        {
            return;
        }
        else if (m =="Join")
        {
            if ( whitePlayer ==NULL_KEY)
            {
                whitePlayer = id;
                llSay(0, llKey2Name(id)+" plays white");

            }
            else if (blackPlayer ==NULL_KEY)
            {
                blackPlayer = id;
                llSay(0, llKey2Name(id)+" plays black");
            }
            
            if (whitePlayer != NULL_KEY && blackPlayer != NULL_KEY)
            {
                llSay(0, "Game starts! White plays ... ");
                sound("hit_open");
                currentColor = WHITE;
                otherColor   = BLACK;
                initBoard();
            }
            else 
                sound("lap");
            refreshBoard();
        }
        else if (m == "Queenside"  || m == "Kingside")
        {
            if ( (currentColor == WHITE && id != whitePlayer) || (currentColor == BLACK && id != blackPlayer))
            {
                llInstantMessage(id, "Not your turn");
                return;
            }
            if (doCastle(m))
            {
                llSay(0, colorName(currentColor)+" castles "+m);
                switchSides();
                refreshBoard();
            }

        }
        else if (m == "Castle...")
        {
            list opts = ["CLOSE", "Queenside", "Kingside"];
            llDialog(id, "Castle: ", opts, chan(llGetKey()));
        }
        else if (m == "Checkmate")
        {
            if (id == currentPlayer)
                llInstantMessage(id, "It is your turn, just capture the king");
            else
            {
                waitUserToResign = currentPlayer;
                llSay(0, llKey2Name(id)+" claims checkmate! You have one minute to make a valid move, or else you will be automatically resigned!");
                llSetTimerEvent(60);
            }   
        }
        else if (m == "Resign")
        {
            llSay(0, llKey2Name(id)+" resigns");
            resignUser(id);
        }
        else if (m == "Quit")
        {
            if (id == whitePlayer) whitePlayer = NULL_KEY;
            else if (id == blackPlayer) blackPlayer = NULL_KEY;
            sound("hit_kill");
            refreshBoard();
        }
        else if (m == "Help")
        {
            llSay(0, "Read a book");
        }
        else if (pendingPromotion>=0 && id == currentPlayer)
        {
            integer np;
            if (m == "Bishop") np = BISHOP;
            else if (m == "Queen") np = QUEEN;
            else if (m == "Rook") np = ROOK;
            else np = KNIGHT;
            setBoard(pendingPromotion, currentColor|np);
            llSay(0, "Pawn promoted to "+m);
            pendingPromotion = -1;
            switchSides();
            sound("lap");
            refreshBoard();
        }
        
    }


    timer()
    {
        if (waitUserToResign!= NULL_KEY)
        {
            resignUser(waitUserToResign);
            waitUserToResign = NULL_KEY;
        }
        
        checkListen();
        if (listener == -1)
            llSetTimerEvent(0);
    }

}

 