// $Id: speech reader.lsl 2 2009-05-23 12:13:30Z jrn $
// Notecard reader
//
// Reads out notecards in inventory, into local chat. Looks for its
// configuration in a notecard called "Configuration" (which is hidden
// from being read out.
//
// Supports access control in the form of owner-only, group or everyone
// style access, with whitelisting and blacklisting as well. Speech rate
// and range can also be configured from the Configuration notecard.

// Don't change this! It's just here incase LSL changes later.
integer BUTTONS_PER_PAGE = 10;

string CONFIGURATION_NOTECARD_NAME = "Configuration";

integer DEBUG = FALSE;

list NOTECARD_SEPARATORS = ["="];
list NOTECARD_SPACERS = [];

string PAUSE_INSTRUCTION = "[PAUSE]";

// Number of seconds after which to pause animation requests.
float TIMEOUT_ANIMATION_REQUEST = 60.0;

// Global variables below here are just used to hold state.
integer g_CommandChannel;

list g_Whitelist = [];
list g_Blacklist = [];

integer g_AllowEveryone = TRUE;
integer g_AllowGroup = FALSE;

// Name of the animation currently being played on the agent operating the
// notecard reader, so it can be stopped when reading has completed.
string g_RunningAnimation;

// Sensor details for detecting the active agent has left the area.
float g_SensorRange = 30.0; // Metres
float g_SensorRate = 60.0; // Seconds

integer g_Shout = FALSE;

// The user currently controlling this script.
key g_DetectedAgentKey;
string g_DetectedAgentName;

// The name (in inventory) of the notecard being read out.
string g_Notecard;

// Interval, in seconds, between reading notecard lines out.
float g_ReadDelay = 5.0;

// The number of the notecard line last read out. Must be less than
// g_RequestNotecardLine before a new line is read out.
integer g_SaidNotecardLine;

// The next line to be read from the notecard.
string g_QueuedNotecardLine;

// Tracks the progress of the notecard as it is read out.
integer g_RequestNotecardLine;
key g_RequestNotecardKey;

integer g_ListenHandle;

// The index of the lowest numbered notecard to be shown in the current page.
integer g_BaseNotecardIdx = 0;

// Determines if the user can acceess this script, based on the access default
// (g_AllowEveryone and g_AllowGroup), and the whitelist/blacklist.
// Returns TRUE if they can access this script, false otherwise.
integer isAllowed(key detectedKey, string detectedName)
{    
    if (detectedKey == llGetOwner()) {
        return TRUE;
    }
    
    // All whitelist/blacklist entries are stored in lower case, so we need to
    // convert to lose case to ensure no case sensitivity issues.
    string lowerName = llToLower(detectedName);
    
    if (llListFindList(g_Whitelist, [lowerName]) >= 0) {
        return TRUE;
    }
            
    if (llListFindList(g_Blacklist, [lowerName]) >= 0) {
        return FALSE;
    }
            
    if (g_AllowEveryone) {
        return TRUE;
    }
            
    if (g_AllowGroup &&
        llSameGroup(detectedKey)) {
        return TRUE;
    }

    return FALSE;
}

parseConfigurationLine(string data)
{
    list parts;
    string name;
    string value;
    
    if (data == "" ||
        llGetSubString(data, 0, 0) == "#")
    {
        // Comment, ignore
        return;
    }
    
    parts = llParseString2List(data, NOTECARD_SEPARATORS, NOTECARD_SPACERS);
    if (llGetListLength(parts) != 2)
    {
        llOwnerSay("Configuration notecard line \""
            + data + "\" (from \""
            + CONFIGURATION_NOTECARD_NAME + "\") could not be parsed.");
        return;
    }
    
    name = llToLower(llList2String(parts, 0));
    value = llToLower(llList2String(parts, 1));
    
    if (name == "access")
    {
        if (value == "owner")
        {
            g_AllowEveryone = FALSE;
            g_AllowGroup = FALSE;
        }
        else if (value == "group")
        {
            g_AllowEveryone = FALSE;
            g_AllowGroup = TRUE;
        }
        else if (value == "everyone" || value == "all")
        {
            g_AllowEveryone = TRUE;
            g_AllowGroup = TRUE;
        }
        else
        {
            llOwnerSay("Unknown access type \""
                + value + "\"; valid values are \"owner\", \"group\" or \"everyone\".");
        }
    }
    else if (name == "blacklist")
    {
        g_Blacklist += [value];
    }
    else if (name == "read_delay")
    {
        g_ReadDelay = (float)value;
        if (g_ReadDelay < 0.1)
        {
            llOwnerSay("Read rate cannot be below 0.1ms, defaulting to 0.1ms.");
            g_ReadDelay = 0.1;
        }
    }
    else if (name == "shout")
    {
        if (value == "true" ||
            value == "yes" ||
            value == "shout")
        {
            g_Shout = TRUE;
        }
        else if (value == "false" ||
            value == "no")
        {
            g_Shout = FALSE;
        }
        else
        {
            llOwnerSay("Unrecognised shout option \""
                + value + "\", expected \"true\" or \"false\".");
        }
    }
    else if (name == "whitelist")
    {
        g_Whitelist += [value];
    }
    else
    {
        llOwnerSay("Unknown parameter \""
            + name + "\"; valid names are; access, blacklist, read_delay, shout or whitelist.");
    }    
    
    return;
}

showNotecardDialog(key detectedKey)
{
    list buttons = [];
    integer notecardIdx;
    integer notecardPage = g_BaseNotecardIdx / BUTTONS_PER_PAGE;
    integer totalNotecards = llGetInventoryNumber(INVENTORY_NOTECARD);
    integer totalPages = totalNotecards / BUTTONS_PER_PAGE;

    if ((totalPages * BUTTONS_PER_PAGE) < totalNotecards)
    {
        totalPages++;
    }

    notecardIdx = g_BaseNotecardIdx;
    while (notecardIdx < totalNotecards &&
        llGetListLength(buttons) < BUTTONS_PER_PAGE)
    {
        string notecardName = llGetInventoryName(INVENTORY_NOTECARD, notecardIdx);

        if (notecardName != CONFIGURATION_NOTECARD_NAME)
        {
            if (llStringLength(notecardName) > 24)
            {
                notecardName = llGetSubString(notecardName, 0, 23);
            }
            buttons = llListInsertList(buttons, [notecardName], 0);
        }
        notecardIdx++;
    }

    buttons = ["Prev"] + llListInsertList(buttons, ["Next"], 1);
                
    llDialog(detectedKey, "Please select a notecard (p. "
        + (string)(notecardPage + 1) + "/"
        + (string)totalPages + ")", buttons, g_CommandChannel);
}

// The default state sets up the command channel and attempts to read in the
// configuration. After the configuration is loaded, it proceeds to the
// "menu_inactive" state
default
{
    on_rez(integer param)
    {
        llResetScript();
    }
    
    dataserver(key queryID, string data) {
        if (queryID == g_RequestNotecardKey) {
            if (data == EOF) {
            state menu_inactive;
            }
            
            parseConfigurationLine(data);
            g_RequestNotecardKey = llGetNotecardLine(CONFIGURATION_NOTECARD_NAME, ++g_RequestNotecardLine);
        }
    }


    state_entry() {
        integer notecardCount = llGetInventoryNumber(INVENTORY_NOTECARD);
        
        g_CommandChannel = -llFloor(llFrand(2147483647));
        
        if (notecardCount == 0)
        {
            llOwnerSay("I do not have any notecards in inventory to read out!");
            state empty;
        }
            
        if (llGetInventoryType(CONFIGURATION_NOTECARD_NAME) != INVENTORY_NOTECARD ||
            llGetInventoryKey(CONFIGURATION_NOTECARD_NAME) == NULL_KEY) {
            state menu_inactive;
        }

        g_RequestNotecardLine = 0;
        g_RequestNotecardKey = llGetNotecardLine(CONFIGURATION_NOTECARD_NAME, g_RequestNotecardLine);
    }
}

state empty
{
    changed(integer changeType)
    {
        if (changeType & CHANGED_INVENTORY)
        {
            llResetScript();
        }
    }

    on_rez(integer param)
    {
        llResetScript();
    }
}

// Once script configuration has completed, the script enters this state, where
// it waits for an agent to click it. Once it receives a click, it checks the
// agent's access privileges, ignores them if they do not have access, or
// shows the dialog menu and goes in to the "menu_active" state if they do have
// access.
state menu_inactive {
    changed(integer changeType)
    {
        if (changeType & CHANGED_INVENTORY)
        {
            llResetScript();
        }
    }
    
    on_rez(integer param)
    {
        llResetScript();
    }
    
    touch_end(integer numDetected)
    {
        integer detectedIdx;
        
        for (detectedIdx = 0; detectedIdx < numDetected; detectedIdx++)
        {
            g_DetectedAgentKey = llDetectedKey(detectedIdx);
            g_DetectedAgentName = llDetectedName(detectedIdx);
            
            if (isAllowed(g_DetectedAgentKey, g_DetectedAgentName))
            {
                showNotecardDialog(g_DetectedAgentKey);
                state menu_active;
            }
        }
    }
}

// When a valid agent requests the menu, the script enters this state. This
// differs from menu_inactive primarily in that it activates the listener for
// dialog responses, which involves some lag (so we want it inactive most of
// the time). If clicked again by the current agent, it re-sends the dialog.
// If clicked by any other valid agent it tells them who is currently using
// the script.
// From here, normal flow would lead to the "start_reading" state once the
// current agent picks a notecard.
state menu_active {
    changed(integer changeType)
    {
        if (changeType & CHANGED_INVENTORY)
        {
            llResetScript();
        }
    }
    
    state_entry()
    {
        g_ListenHandle = llListen(g_CommandChannel, "", "", "");
        llSetTimerEvent(60.0);
    }
    
    state_exit()
    {
        llListenRemove(g_ListenHandle);
        llSetTimerEvent(0.0);
    }
    
    on_rez(integer param)
    {
        llResetScript();
    }
    
    listen(integer channel, string name, key id, string message)
    {
        if (g_DetectedAgentKey == id)
        {
            if (message == "Next") {
                integer totalNotecards = llGetInventoryNumber(INVENTORY_NOTECARD);

                g_BaseNotecardIdx += BUTTONS_PER_PAGE;
                if (g_BaseNotecardIdx > totalNotecards)
                {
                    g_BaseNotecardIdx = totalNotecards - (totalNotecards % 10);
                    if (g_BaseNotecardIdx == totalNotecards)
                    {
                        g_BaseNotecardIdx -= 10;
                    }
                }
                
                showNotecardDialog(id);
            } else if (message == "Prev") {
                g_BaseNotecardIdx -= BUTTONS_PER_PAGE;
                if (g_BaseNotecardIdx < 0)
                {
                    g_BaseNotecardIdx = 0;
                }
                
                showNotecardDialog(id);
            } else {
                integer notecardIdx;
                integer totalNotecards = llGetInventoryNumber(INVENTORY_NOTECARD);
               
                // If the full name of the notecard fitted into the dialog button, which is the normal case,
                // we can take something of a short-cut here. 
                if (llGetInventoryType(message) == INVENTORY_NOTECARD) {
                    if (llGetInventoryKey(message) == NULL_KEY) {
                        llInstantMessage(id, "Notecard \""
                            + message + "\" is empty!");
                        state menu_inactive;
                    }
                    
                    g_Notecard = message;
                    state start_reading;
                }
                
                for (notecardIdx = 0; notecardIdx < totalNotecards; notecardIdx++) { // For loop to count through all the objects.
                    string inventoryName = llGetInventoryName(INVENTORY_NOTECARD, notecardIdx);
                
                    if (llSubStringIndex(inventoryName, message) != -1) {
                        // If the word the person says is contained within an inventory item...
                        if (llGetInventoryKey(inventoryName) == NULL_KEY) {
                            llInstantMessage(id, "Notecard \""
                                + inventoryName + "\" is empty!");
                            state menu_inactive;
                        }
                        g_Notecard = inventoryName;
                        state start_reading;
                    }
                }
            }
        }
        else if (isAllowed(id, name))
        {
            llInstantMessage(id, "I am currently in use by \""
                + g_DetectedAgentName + "\".");
        }
    }
    
    timer()
    {
        llInstantMessage(g_DetectedAgentKey, "Dialog timed out after 60 seconds.");
        state menu_inactive;
    }
    
    touch_end(integer detectedCount)
    {
        integer detectedIdx;
        
        for (detectedIdx = 0; detectedIdx < detectedCount; detectedIdx++) {
            key currentDetectedKey = llDetectedKey(detectedIdx);
            
            if (currentDetectedKey == g_DetectedAgentKey) {
                showNotecardDialog(g_DetectedAgentKey);
            }
            else if (isAllowed(currentDetectedKey, llDetectedName(detectedIdx)))
            {
                llInstantMessage(currentDetectedKey, "I am currently in use by \""
                    + g_DetectedAgentName + "\".");
            }
        }
    }
}

// Sets up the script for the new notecard. If there is an animation in prim
// inventory, requests animation permissions from the avatar. Otherwise, it
// simply starts reading anyway. After TIMEOUT_ANIMATION_REQUEST seconds it
// gives up waiting for animation permissions and continues without them.
state start_reading
{
    changed(integer changeType)
    {
        if (changeType & (CHANGED_INVENTORY | CHANGED_OWNER))
        {
            llResetScript();
        }
    }
    
    on_rez(integer param)
    {
        llOwnerSay("Current waiting on animation permissions from "
            + g_DetectedAgentName + "; will auto-reset within "
            + (string)g_SensorRate + " seconds if they are not present.");
    }
    
    run_time_permissions(integer perm)
    {
        if (perm & PERMISSION_TRIGGER_ANIMATION)
        {
            state reading;
        }
    }
    
    no_sensor()
    {
        llInstantMessage(g_DetectedAgentKey, "Resetting speech reader because you have left the area.");
        llResetScript();
    }
    
    sensor(integer detectedCount)
    {
        // Ignore; this is just here to make no_sensor work.
    }

    state_entry()
    {
        // If the reading agent leaves the area, we need to know so we reset.
        llSensorRepeat("", g_DetectedAgentKey, AGENT, g_SensorRange, PI, g_SensorRate);
        
        llSetTimerEvent(TIMEOUT_ANIMATION_REQUEST);
        
        llInstantMessage(g_DetectedAgentKey, "Reading from notecard \""
            + g_Notecard + "\"; touch me again to stop reading before the end of the notecard.");

        g_RequestNotecardLine = 0;
        
        if (llGetInventoryNumber(INVENTORY_ANIMATION) == 0)
        {
            state reading;
        }
        
        g_RunningAnimation = llGetInventoryName(INVENTORY_ANIMATION, 0);
        llRequestPermissions(g_DetectedAgentKey, PERMISSION_TRIGGER_ANIMATION);
    }
    
    state_exit()
    {
        llSetTimerEvent(0.0);
    }
    
    timer()
    {
        state reading;
    }
}

state reading
{
    changed(integer changeType)
    {
        if (changeType & (CHANGED_INVENTORY | CHANGED_OWNER))
        {
            if (DEBUG)
            {
                llOwnerSay("Resetting due to owner or inventory change.");
            }
            llResetScript();
        }
    }
    
    on_rez(integer param)
    {
        llOwnerSay("Current reading from the notecard \""
            + g_Notecard + "\" for "
            + g_DetectedAgentName + "; will auto-reset within "
            + (string)g_SensorRate + " seconds if they are not present.");
    }
    
    dataserver(key queryID, string data)
    {
        if (queryID == g_RequestNotecardKey)
        {
            g_RequestNotecardLine++;
            
            if (data == EOF)
            {
                llInstantMessage(g_DetectedAgentKey, "Reached end of notecard "
                    + g_Notecard + ", resetting script.");
                llResetScript();
            }
            else if (llStringTrim(data, STRING_TRIM) == PAUSE_INSTRUCTION)
            {
                state paused;
            }
            else
            {
                g_QueuedNotecardLine = data;
            }
        }
        else
        {
            llOwnerSay("Received unexpected data server event "
                + (string)queryID + ".");
        }
    }
    
    no_sensor()
    {
        llInstantMessage(g_DetectedAgentKey, "Resetting speech reader because you have left the area.");
        llResetScript();
    }
    
    sensor(integer detectedCount)
    {
        // Ignore; this is just here to make no_sensor work.
    }

    state_entry()
    {
        if (DEBUG)
        {
            llOwnerSay("Entered reading state");
        }
        
        if (llGetPermissions() & PERMISSION_TRIGGER_ANIMATION)
        {
            llStartAnimation(g_RunningAnimation);
        }
        
        llSetTimerEvent(g_ReadDelay);
        g_RequestNotecardKey = llGetNotecardLine(g_Notecard, g_RequestNotecardLine);
    }
    
    state_exit()
    {
        if (DEBUG)
        {
            llOwnerSay("Leaving reading state");
        }
        
        if (llGetPermissions() & PERMISSION_TRIGGER_ANIMATION)
        {
            llStopAnimation(g_RunningAnimation);
        }
        
        llSetTimerEvent(0.0);
    }
    
    timer()
    {        
        if (g_SaidNotecardLine < g_RequestNotecardLine)
        {
            if (g_Shout)
            {
                llShout(0, g_QueuedNotecardLine);
            }
            else
            {
                llSay(0, g_QueuedNotecardLine);
            }
            
            g_RequestNotecardKey = llGetNotecardLine(g_Notecard, g_RequestNotecardLine);
        }
    }
    
    touch_end(integer touchCount)
    {
        integer touchIdx;
        
        for (touchIdx = 0; touchIdx < touchCount; touchIdx++)
        {
            if (llDetectedKey(touchIdx) == g_DetectedAgentKey)
            {
                state menu_inactive;
            }
        }
    }
}

state paused
{
    changed(integer changeType)
    {
        if (changeType & (CHANGED_INVENTORY | CHANGED_OWNER))
        {
            llResetScript();
        }
    }
    
    on_rez(integer param)
    {
        llOwnerSay("Current paused by "
            + g_DetectedAgentName + "; will auto-reset within "
            + (string)g_SensorRate + " seconds if they are not present.");
    }
    
    no_sensor()
    {
        llInstantMessage(g_DetectedAgentKey, "Resetting speech reader because you have left the area.");
        llResetScript();
    }
    
    sensor(integer detectedCount)
    {
        // Ignore; this is just here to make no_sensor work.
    }
    
    state_entry()
    {
        if (DEBUG)
        {
            llOwnerSay("Entered paused state");
        }
        
        llInstantMessage(g_DetectedAgentKey, "Paused; click me to restart playback. Script will auto-reset if you leave the area.");
    }
    
    touch_end(integer detectedCount)
    {
        integer detectedIdx;
        
        for (detectedIdx = 0; detectedIdx < detectedCount; detectedIdx++)
        {
            key currentKey = llDetectedKey(detectedIdx);
            
            if (currentKey == g_DetectedAgentKey)
            {
                state reading;
            }
            else if (isAllowed(currentKey, llDetectedName(detectedIdx)))
            {
                llInstantMessage(currentKey, "Playback is currently paused by "
                    + g_DetectedAgentName + ".");
            }
        }
    }
}
