// Script Name: RealFire.lsl
// Author: Rene
//Realistic fire with smoke, light and sound support

// Downloaded from : http://www.free-lsl-scripts.com/freescripts.plx?ID=1598

// This program is free software; you can redistribute it and/or modify it.
// License information must be included in any script you give out or use.
// This script is licensed under the Creative Commons Attribution-Share Alike 3.0 License
// from http://creativecommons.org/licenses/by-sa/3.0 unless licenses are
// included in the script or comments by the original author,in which case
// the authors license must be followed.

// Please leave any authors credits intact in any script you use or publish.
////////////////////////////////////////////////////////////////////
// Realistic fire with smoke, light and sound support
//
// Author: Rene Schrama
// Date: 01-07-2011
// Project content: fire.lsl, smoke.lsl
//
// THIS SCRIPT WAS DEVELOPED IN OPENSIM AND CANNOT BE GUARANTEED TO WORK ANYWHERE ELSE. USE AT YOUR OWN RISK.
//
// This work is licensed under the Creative Commons Attribution-ShareAlike 3.0 Unported (CC BY-SA 3.0) License.
// To view a copy of this license, visit http://creativecommons.org/licenses/by-sa/3.0/.
//
// - Produces fire, smoke or both
// - Touch to start or stop fire
// - Long touch to show or hide smoke
// - Light and (optional) sound (drop this in the same prim)
// - Burns down (see Constants below)
// - Based on Ama Omega's Particle Script 0.5 (26-03-2004)
//
// How to use: every prim can only have ONE particle emitter, so you cannot drop this script in the same
// prim as the smoke script. You may want to drop it in the root prim and the smoke script in a child prim.
// Note: it is assumed that you have some sort of object (linkset) with at least 2 prims.
// Recommended environment setting: Midnight.

string title = "RealFire";      // Title
string version = "1.0";         // Version

// ==================================================================================================
// Constants (change to suit your own taste)

float time = 5;                 // Timer interval in seconds
integer burnTime = 60;          // Number of timer intervals to burn before starting to die
float decPercent = 1;           // How much to burn down (%) every timer interval
float minPercent = 25;          // When to stop the fire (%)

// At the current settings, the fire will burn for 5 minutes, then it will die in just over 6 minutes
// ==================================================================================================

// Mask Flags - set to TRUE to enable

integer glow = TRUE;            // Make the particles glow
integer bounce = FALSE;         // Do not make particles bounce on Z plane of object
integer interpColor = TRUE;     // Go from start to end color
integer interpSize = TRUE;      // Go from start to end size
integer wind = FALSE;           // Particles not affected by wind
integer followSource = FALSE;   // Particles do not follow the source
integer followVel = TRUE;       // Particles turn to velocity direction

// Pattern

integer pattern = PSYS_SRC_PATTERN_EXPLODE;

// Particle parameters

float age = 1.0;                // Life of each particle
float minSpeed = 0.0;           // Min speed each particle is spit out at
float maxSpeed = 0.01;          // Max speed each particle is spit out at
string texture = "";            // Texture used for particles, default used if blank
float startAlpha = 1.0;         // Start alpha (transparency) value
float endAlpha = 0.0;           // End alpha (transparency) value
vector startColor = <1,1,0>;    // Start color of particles <R,G,B> (fire = yellow to red)
vector endColor = <1,0,0>;      // End color of particles <R,G,B> (if interpColor == TRUE)
vector startSize = <0.2,0.5,0>; // Start size of particles
vector endSize = <0.2,0.5,0>;   // End size of particles (if interpSize == TRUE)
vector push = <0,0,2.5>;        // Force pushed on particles

// System parameters

float rate = 0.1;               // How fast (rate) to emit particles
float radius = 0.1;             // Radius to emit particles for BURST pattern
integer count = 10;             // How many particles to emit per BURST
float life = 0.0;               // Life in seconds for the system to make particles

// Script variables

float scriptTime = 0.0;         // script time
integer smoke = TRUE;           // show or hide smoke
integer flags = 0;              // flags
list sys = [];                  // particle system
integer counter = 0;            // this will count the number of timer intervals
float percent = 0.0;            // percentage of particle lifetime
float percentLight = 0.0;       // percentage of light and sound
integer percentLightInt = 0;    // rounded percentage of light and sound
float decPercentLight = 0.0;    // how much to decrease percentLight
vector lightColor = <1,1,1>;    // Light color
float lightIntensity = 1.0;     // light intensity
float lightRadius = 10.0;       // light radius
float volume = 1.0;             // sound volume
string sound = "";              // first sound in inventory

// Original values

float _age;                     // Life of each particle
float _lightIntensity;          // light intensity
float _lightRadius;             // light radius
float _volume;                  // sound volume

// Functions

float percentage (float per, float num)
{
    return num / 100.0 * per;
}

startSystem()
{
    percent = 100.0;
    percentLight = 100.0;
    age = _age;
    lightIntensity = _lightIntensity;
    lightRadius = _lightRadius;
    sound = llGetInventoryName(INVENTORY_SOUND, 0); // get first sound from inventory
    volume = _volume;
    updateParticles();
    if (sound) llLoopSound(sound, volume);
    counter = 0;
    llSetTimerEvent(0.0);
    llSetTimerEvent(time);
    if (smoke) llMessageLinked(LINK_ALL_OTHERS, TRUE, "", "");
}

stopSystem()
{
    percent = 0.0;
    llSetTimerEvent(0.0);
    llParticleSystem([]);
    llSetPrimitiveParams([PRIM_POINT_LIGHT, FALSE, <0,0,0>, 0, 0, 0]);
    if (sound) llStopSound();
    if (smoke) llMessageLinked(LINK_ALL_OTHERS, FALSE, "", "");
}

updateParticles()
{
    sys = [ PSYS_PART_MAX_AGE, age,
        PSYS_PART_FLAGS, flags,
        PSYS_PART_START_COLOR, startColor,
        PSYS_PART_END_COLOR, endColor,
        PSYS_PART_START_SCALE, startSize,
        PSYS_PART_END_SCALE, endSize,
        PSYS_SRC_PATTERN, pattern,
        PSYS_SRC_BURST_RATE, rate,
        PSYS_SRC_ACCEL, push,
        PSYS_SRC_BURST_PART_COUNT, count,
        PSYS_SRC_BURST_RADIUS, radius,
        PSYS_SRC_BURST_SPEED_MIN, minSpeed,
        PSYS_SRC_BURST_SPEED_MAX, maxSpeed,
        PSYS_SRC_MAX_AGE, life,
        PSYS_SRC_TEXTURE, texture,
        PSYS_PART_START_ALPHA, startAlpha,
        PSYS_PART_END_ALPHA, endAlpha ];
    llParticleSystem(sys);
    llSetPrimitiveParams([ PRIM_POINT_LIGHT, TRUE,
        lightColor,        // light color vector range: 0.0-1.0
        lightIntensity,    // intensity   (0.0-1.0)
        lightRadius,       // radius      (0.1-20.0)
        0.75 ]);           // falloff     (0.1-2.0)
    if (sound) llAdjustSoundVolume(volume);
    percentLightInt = llRound(percentLight);
    if (smoke && percentLightInt > 1) llMessageLinked(LINK_ALL_OTHERS, percentLightInt, "", "");
}

default
{
    state_entry()
    {
        llParticleSystem([]);
        _age = age;
        _lightIntensity = lightIntensity;
        _lightRadius = lightRadius;
        _volume = volume;
        lightColor = (startColor + endColor) / 2.0; // light color = average of start & end color
        decPercentLight = 100.0 / (100.0 - minPercent) * decPercent; // ligtht & sound always go all the way down
        if (glow) flags = flags | PSYS_PART_EMISSIVE_MASK;
        if (bounce) flags = flags | PSYS_PART_BOUNCE_MASK;
        if (interpColor) flags = flags | PSYS_PART_INTERP_COLOR_MASK;
        if (interpSize) flags = flags | PSYS_PART_INTERP_SCALE_MASK;
        if (wind) flags = flags | PSYS_PART_WIND_MASK;
        if (followSource) flags = flags | PSYS_PART_FOLLOW_SRC_MASK;
        if (followVel) flags = flags | PSYS_PART_FOLLOW_VELOCITY_MASK;
        llWhisper(0, title + " " + version + " ready.");
        llWhisper(0, "Particle count: " + (string)llRound(1.0 / rate * (float)count * age) + ".");
        llWhisper(0, "Free memory: " + (string)llGetFreeMemory() + " bytes.");
        llWhisper(0, "Touch to start/stop fire.");
        llWhisper(0, "Long touch to show/hide smoke.");
    }

    on_rez(integer start_param)
    {
        llResetScript();
    }

    timer()
    {
        counter++;
        if (counter < burnTime) return;
        if (percent >= minPercent + decPercent) {
            percent -= decPercent;
            percentLight -= decPercentLight;
            age = percentage(percent, _age);
            lightIntensity = percentage(percentLight, _lightIntensity);
            lightRadius = percentage(percentLight, _lightRadius);
            volume = percentage(percentLight, _volume);
            updateParticles();
        }
        else {
            stopSystem();
        }
    }

    touch_start(integer total_number)
    {
        scriptTime = llGetTime();
    }

    touch_end(integer total_number)
    {
        if (llGetTime() - scriptTime > 1.0) {
            if (smoke) {
                if (percent > 0.0) llWhisper(0, "Smoke OFF.");
                else llWhisper(0, "Smoke OFF (will be applied to next fire).");
                llMessageLinked(LINK_ALL_OTHERS, FALSE, "", "");
                smoke = FALSE;
            }
            else {
                if (percent > 0.0) llWhisper(0, "Smoke ON.");
                else llWhisper(0, "Smoke ON (no fire).");
                llMessageLinked(LINK_ALL_OTHERS, TRUE, "", "");
                smoke = TRUE;
            }
        }
        else {
            if (percent > 0.0) stopSystem();
            else startSystem();
        }
    }
}
 