// Program written by Grandma Bates
//
// It is used to operate a door
//
// The values used for the door are read from the notecard "Door Values"
// It is licensed under a BSD license which means that you are
// free to do what you want with it. You need only give Grandma Bates
// some credit for her work.
// 
//######################################################################
// Copyright (c) 2007, Grandma Bates (contact via IM within Second Life)
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
//    * Redistributions of source code must retain the above copyright
//      notice, this list of conditions and the following disclaimer.
//
//    * Redistributions in binary form must reproduce the above
//      copyright notice, this list of conditions and the following
//      disclaimer in the documentation and/or other materials provided
//      with the distribution.
//
//    * Neither the name of Grandma Bates nor the names of its
//      contributors may be used to endorse or promote products derived
//      from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
// FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
// COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
// BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
// ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// (This copyright notice was copied from the BSD license found at
//  http://www.opensource.org/licenses/bsd-license.php.)
//
//######################################################################



// The default values are specified below. 
// Please note that these get overridden by reading from the 
// notecard, "Door Values."
float angle = 90.0;
integer numAngles = 20;
float timeOpen = 15.0;
integer closeAutomatically;

// Variables necessary for door operation.
rotation baseRot;
vector basePos;
vector scale;


// Keys for reading notecards
key readAngle;
key readNumAngle;
key readTime;
key readShut;

// variables indicating the door states
integer open;
integer changingDoor;

// Variables for listening for the close command
integer channel;
integer listenControl;

toggleDoor() {
    integer doorSwing;
    rotation rotThrough;
    rotation theBase;

    if(changingDoor) {
        llSay(0,"The door is currently in operation. Please try again shortly.");
        return;
    }

    //llSay(0,(string)llGetLinkNumber());
    if( (llGetLinkNumber() != LINK_ROOT) && ( llGetLinkNumber() != 0) ) {
        theBase = llGetRootRotation();
    } else {
        theBase = ZERO_ROTATION;
    }
    changingDoor = TRUE;
    if (open) {
        // close the door.
        open = FALSE;
        for (doorSwing=numAngles;doorSwing>=0;--doorSwing) {
            rotThrough = llEuler2Rot(
               <0.0,0.0,angle*PI/180.0*((float)doorSwing)/((float)numAngles)>);
            llSetPrimitiveParams([PRIM_ROTATION,rotThrough*baseRot/theBase,
                PRIM_POSITION,basePos + <0.5*scale.x,0.0,0.0>*rotThrough*baseRot -
                    <0.5*scale.x,0.0,0.0>*baseRot]);
            //llSetLocalRot(rotThrough*baseRot);
            //llSetPos(basePos + <0.5*scale.x,0.0,0.0>*rotThrough*baseRot
            //                      - <0.5*scale.x,0.0,0.0>);
        }
        llSetLocalRot(baseRot);
        llSetPos(basePos);
    } else {
        // open the door.
        open = TRUE;
        for (doorSwing=0;doorSwing<=numAngles;++doorSwing) {
            rotThrough = llEuler2Rot(
                <0.0,0.0,angle*PI/180.0*((float)doorSwing)/((float)numAngles)>);
            llSetPrimitiveParams([PRIM_ROTATION,rotThrough*baseRot/theBase,
                PRIM_POSITION,basePos + <0.5*scale.x,0.0,0.0>*rotThrough*baseRot -
                    <0.5*scale.x,0.0,0.0>*baseRot]);
            //llSetLocalRot(rotThrough*baseRot);
            //llSetPos(basePos + <0.5*scale.x,0.0,0.0>*rotThrough*baseRot
            //                      - <0.5*scale.x,0.0,0.0>);
        }
    }
    changingDoor = FALSE;

}




default
{
    state_entry(){


        // Request to read the values from the notecard.
        readAngle    = llGetNotecardLine("Door Values",1);
        readNumAngle = llGetNotecardLine("Door Values",3);
        readTime     = llGetNotecardLine("Door Values",5);
        readShut     = llGetNotecardLine("Door Values",7);

        baseRot = llGetLocalRot();
        basePos = llGetLocalPos();
        scale = llGetScale();
        open = FALSE;
        changingDoor = FALSE;
        closeAutomatically = TRUE;
        
        // Listen for the "close" command from my owner
        channel = 2;
        listenControl = llListen(channel,"",llGetOwner(),"close");
        llOwnerSay("If you say \"/"+(string)channel+" close\" then I will automatically close.");

    }

    // Make sure I reset myself when rezzed
    on_rez(integer param) {
        llResetScript();
    }

    touch_start(integer num_detected) {
        integer lupe;

         if(!open && (!changingDoor) ) {
            baseRot = llGetLocalRot();
            basePos = llGetLocalPos();
            scale = llGetScale();
            toggleDoor();
            // wait to close the door
            if(closeAutomatically) {
                llSetTimerEvent(timeOpen);
            }
        } else if ( !closeAutomatically && open && (!changingDoor)  ) {
            toggleDoor();
        } else {
            llSay(0,"Sorry I am already open. Please wait until I close");
        }

    }


    listen(integer channel, string name, key id, string message) {
        // There is only one listen control
        // only the owner saying "close" will get me here.
        if ( !closeAutomatically && open && (!changingDoor)  ) {
            toggleDoor();
        } 


    }   


    timer() {
        // close the door as long as I am not in operation
        if(open && (!changingDoor) ) {
            toggleDoor();
            llSetTimerEvent(0.0);
        }
    }

        
    // Routine to read the values from the notecard.
    dataserver(key query_id, string data) {
        if (readAngle == query_id) {
            // read in the angle to open to
            angle = (float) data;
            llOwnerSay("The angle that the door will turn is " +
                       (string) angle);
            readAngle = NULL_KEY;
        } else if (readNumAngle == query_id) {
            // read in the number of steps to take
            numAngles = (integer) data;
            llOwnerSay("The number of increments it turns is " + (string)numAngles);
            readNumAngle = NULL_KEY;
        } else if (readTime == query_id) {
            // read in how long the door will be open
            timeOpen = (float) data;
            llOwnerSay("The time that the door is open is " + (string)timeOpen);
            readTime = NULL_KEY;
        } else if (readShut == query_id) {
            closeAutomatically = (llToUpper(data) == "YES");
            if(closeAutomatically) {
                llOwnerSay("I will shut automatically");
            } else {
                llOwnerSay("I will not shut automatically.");
            }
            readShut = NULL_KEY;
        }
            
            
    }

    // If the inventory has changed re-read the door parameters
    changed(integer change) { 
        if(change & CHANGED_INVENTORY) {
            // Request to read the values from the notecard.
            readAngle    = llGetNotecardLine("Door Values",1);
            readNumAngle = llGetNotecardLine("Door Values",3);
            readTime     = llGetNotecardLine("Door Values",5);
            readShut     = llGetNotecardLine("Door Values",7);
        }
    }
         
}