//XEngine:lsl
//Simple door
//
//Features:
//Works in Open Simulator based grids even when door is linked to other prims.
//Does not require a hinge prim. See instructions for setting up door prim.
//Optional playing of sounds when the door opens or closes.
//Optional automatic closing of door using a timer.
//
//This script automatically adjusts for changes in position or rotation
//provided the changes occur while the door is in the closed position.
//
//This script can be use both with, and without, a hinge prim.
//To avoid need for a hinge prim, create the door prim as follows:
//1. Under "Size (meters)" on Object tab, X is the door thickness, Y is
//   *twice* the desired width of the door, and Z is the height of the door.
//2. Under "Path Cut Being and End" set B to 0.375, and E to 0.875
//
//Modified June 10, 2009 by Andrew Hellershanks
//Modified to work in an Open Simulator based grid.
//
//Modified July 3, 2009 by Andrew Hellershanks
//Simplified script by eliminating use of a second state.

// ********** USER SETTINGS HERE **********

//Direction of door rotation. Either 1 (outwards) or -1 (inwards);
integer DIRECTION   = 1;

//These variables indicate if a sound file is to be played when the door
//is opened or closed. If the string is a name, the sound file needs to
//be found in the contents of the door prim. If the string is a key, any
//valid UUID for a sound file may be used. To disable playing sounds when
//the door opens or closes, use the empty string "".
string  SOUND_OPEN  = "doorOpen"; //Sound when door opens
string  SOUND_CLOSE = "doorClose"; //Sound when door closes

//Automatically close the door after this many seconds.
//Use a value of 0 to stop the door from closing automatically.
float   TIMER_CLOSE = 10.0;

// ********** END OF USER SETTINGS **********


integer DOOR_CLOSE  = 0;
integer DOOR_OPEN   = 1;

integer is_open     = FALSE;    

//In Second Life objects can move slightly from their position when rotated.
//This will keep track of the doors position to prevent it from moving.
vector  mypos;

door(integer action)
{
    rotation rot;
    rotation delta;
    
    if (action == DOOR_OPEN)
    {
        //Touching door twice in rapid succession makes it open more than once.
        //Prevent multiple rotations by checking to see if door is already open.
        if (is_open)
            return;

        is_open = TRUE;

        if (SOUND_OPEN != "")
            llTriggerSound(SOUND_OPEN, 0.8);
       
        mypos = llGetLocalPos();    //Remember original position of door

        rot = llGetLocalRot();
        delta = llEuler2Rot(<0, 0, -DIRECTION * PI_BY_TWO>);
        rot = delta * rot;  //Rotate by -45 degree
        llSetLocalRot(rot);

        if (TIMER_CLOSE > 0)
            llSetTimerEvent(TIMER_CLOSE);
    }
    else
    {
        //Touching door twice in rapid succession makes it close more than once.
        //Prevent multiple rotations by checking to see if door is already open.
        if (!is_open)
            return;

        if (TIMER_CLOSE > 0)
            llSetTimerEvent(0);

        is_open = FALSE;

        rot = llGetLocalRot();
        delta = llEuler2Rot(<0, 0, DIRECTION * PI_BY_TWO>);
        rot = delta * rot;  //Rotate by 45 degree
        llSetLocalRot(rot);

        llSetPos(mypos);    //Workaround for tiny movements during rotation

        if (SOUND_CLOSE != "")
            llTriggerSound(SOUND_CLOSE, 0.8);
    }
}

default
{
    on_rez(integer start_params)
    {
        llResetScript();
    }

    touch_start(integer total_number)
    {
        if (is_open)
            door(DOOR_CLOSE);
        else
            door(DOOR_OPEN);
    }

    collision(integer number)
    {
        door(DOOR_OPEN);
    }

    timer() // Time to close the door
    {
        door(DOOR_CLOSE);
    }
}