
// MLPV2.2 Add-On example to support xcite arousal.
// Easily enhanced for other xcite features.

// LMs:
//
//  num = 0 and msg = "POSEB": new pose
//    id = pose name
//
//  num = -11000: avatar sits
//    msg = ballnumber|anim_name   (ballnumber is 0,1,2, or 3)
//    id = avatar who sat
//
//  num = -11001: avatar unsits
//    msg = ballnumber
//    id = avatar who stood
//
//  num = -11002: avatar anim change
//    msg = ballnumber|anim_name
//    id = avatar who is sitting
//
// This script does not use the anim_name parameter or the -11002 LM.

// debugs:
// 1                 = av sit/unsit
// 2                 = setup pose
// 4                 = start anim
// 8                 = get LM from MLP
// 0x10              = send LM to xcite

// internal variables

integer MAX_AVS     = 6;

integer BallCount;

string  Pose;
list    Avnames;    // one for each ball

// List of avatar names per ball: adding/removing

addAv(string name, integer ballIx) {
    Avnames = llListReplaceList(Avnames, (list)name, ballIx, ballIx);
}

removeAv(integer ballIx) {
    addAv("", ballIx);
}


xcite(integer num, string msg) {
    debug(0x10, "lm to xc: " + (string) num + ", '" + msg + "'");
    llMessageLinked(LINK_THIS, num, msg, "");
}


// Card-configurables:

float   Interval;
integer Debug;

list    AnimNames;
list    AnimCommands;

integer PoseCfgCount;

list    PoseConfigs;
list    Amounts;
list    Maxes;
list    Kinks;
list    Tilts;

config_init()
{
    PoseCfgCount    = 0;
    PoseConfigs     = [];
    Amounts         = [];
    Maxes           = [];
    Kinks           = [];
    Tilts           = [];
    Debug           = 0;
    Interval        = 15.0;

    AnimNames       = [];
    AnimCommands    = [];

}


config_parse(string str, string cardName, integer lineNum)
{
    str = llStringTrim(str, STRING_TRIM_HEAD);  // delete leading spaces

    // lines beginning with slash are comments -- ignore them
    if (llGetSubString(str,0,0) == "/") {
        return;
    }

    list ldata      = llParseStringKeepNulls(str, ["  =  ","  = "," =  "," = "," =","= ","="], [""]);
    string cmd      = llList2String(ldata,0);
    string arg1     = llList2String(ldata,1);
    string arg2     = llList2String(ldata,2);

    if (cmd == "pose") {
        PoseCfgCount += 1;
        string  pose    = arg1;
        string  ball    = arg2;
        string  arousal = llList2String(ldata,3);
        string  max     = llList2String(ldata,4);
        string  kink    = llList2String(ldata,5);
        string  tilt    = llList2String(ldata,6);

        if (ball == "*") {
            integer ix;
            for (ix = 0; ix < MAX_AVS; ++ix) {
                addConfig(pose + "|" + (string)ix, arousal, max, kink, tilt);
            }
        } else {
                addConfig(pose + "|" + ball, arousal, max, kink, tilt);
        }
    } else if (cmd == "timer") {
            Interval    = (float) arg1;
    } else if (cmd == "anim") {
            if (arg1 != "" && arg2 != "") {
                debug(0x20, arg1 + " = " + arg2);
                if (llGetInventoryType(arg1) != INVENTORY_ANIMATION) {
                    llWhisper(0, "Warning: anim not found in object inventory: " + arg1);
                }
                AnimNames += (list) arg1;
                AnimCommands += (list) arg2;
            }
    }

    else if (cmd == "debug") {
        Debug = Debug | (integer) arg1;
    }
}


// Post-process any config, if necessary
config_done() {
    announce();
}


addConfig(string ref, string arousal, string max, string kink, string tilt) {
    PoseConfigs   += (list)ref;
    Amounts     += (list)arousal;
    Maxes       += (list)max;
    Kinks       += (list)kink;
    Tilts       += (list)tilt;
}

doTilt(string avname, integer bix) {
    integer cix = llListFindList(PoseConfigs, (list)(Pose + "|" + (string)bix));
    if (cix == -1) {
        xcite(20014, avname);
        return;
    }

    string tilt = llList2String(Tilts, cix);
    if (tilt == "") {
        xcite(20014, avname);
    } else {
            xcite(20020, avname + "|" + tilt);
    }
}

setPose(string pose) {
    Pose = pose;
    integer bix;
    string  avname;

    debug(2, "Pose: " + pose);

    for (bix = 0; bix < BallCount; ++bix) {
        avname = llList2String(Avnames, bix);
        if (avname != "") {
            doTilt(avname, bix);
        }
    }
}

tickle(string av, integer ix) {
    string amount = llList2String(Amounts, ix);
    if (amount != "") {
        string txt = av
            + "|"  + amount
            + "|"  + llList2String(Maxes, ix)
            + "||" + llList2String(Kinks, ix);
        xcite(20001, txt);
    }
}

new_anim(key id, integer ballnum, string anim) {
    string avname = llKey2Name(id);
    integer ix = llListFindList(AnimNames, (list)anim);

    debug(4, "anim " + anim + " on ball " + (string)ballnum);

    if (ix != -1 && avname != "") {
        xcite(20001, avname + "|" + customize(llList2String(AnimCommands, ix)));
    }
}



// In 'src', replace 'target' string with 'replacement' string,
// and return the result.

string replace_text(string src, string target, string replacement)
{
    string  text = src;
    integer ix;

    ix = llSubStringIndex(src, target);

    if (ix >= 0) {
        text = llDeleteSubString(text, ix, ix + llStringLength(target) - 1);
        text = llInsertString(text, ix, replacement);
    }

    return (text);
}

string firstname(string name) {
    return llList2String(llParseString2List(name, [" "], []), 0);
}

string customize(string src) {
    integer avix;
    string avname;
    for (avix = 0; avix < MAX_AVS; ++avix) {
        avname = firstname(llList2String(Avnames, avix));
        if (avname == "") avname = "somebody";
        src = replace_text(src, "/"+(string)avix, avname);
    }
    return src;
}


string plural(string singular, string plural, integer count) {
    if (count != 1) {
        return plural;
    }
    return singular;
}

announce()
{
    integer acount = llGetListLength(AnimNames);
    integer pcount = llGetListLength(PoseConfigs);
    llOwnerSay((string)acount
        + " xcite "
        + plural("anim", "anims", acount)
        + " and "
        + (string) PoseCfgCount
        + plural(" pose", " poses", PoseCfgCount)
        + " configured ("
        + llGetScriptName()
        + ": "
        + (string)llGetFreeMemory()
        + " bytes free)");
}

// ==== Utilities ====

say(string str)
{
    llOwnerSay(str);
}

debug(integer mask, string txt) {
    if (mask & Debug) {
        llWhisper(0, llGetScriptName() + ": " + txt);
    }
}


// Static parameters for reading card config
string  ConfigNotecardPrefix    = ".XCITE";     // string identifying config notecards
float   ConfigTimeout           = 60.0;         // seconds per card to wait for slow server

// Globals for reading card config
integer ConfigLineIndex;
key     ConfigRequestID;
list    ConfigCards;        // list of names of config cards
string  ConfigCardName;     // name of card being read
integer ConfigCardIndex;    // index of next card to read
string  ConfigCardKeys;     // for detecting config change

// return TRUE if there is a config card

integer next_card()
{
    if (ConfigCardIndex >= llGetListLength(ConfigCards)) {
        return (FALSE);
    }

    ConfigLineIndex = 0;
    ConfigCardName = llList2String(ConfigCards, ConfigCardIndex);
    ConfigCardIndex++;
    ConfigRequestID = llGetNotecardLine(ConfigCardName, ConfigLineIndex);
    say("Reading " + ConfigCardName);
    return (TRUE);
}

// Get list of config cards, and return list of their keys for use
// in detecting a config change.

string get_cards() {
    string item;
    string keys;

    ConfigCards = [];
    integer n = llGetInventoryNumber(INVENTORY_NOTECARD);
    while (n-- > 0) {
        item = llGetInventoryName(INVENTORY_NOTECARD, n);
        if (llSubStringIndex(item, ConfigNotecardPrefix) != -1) {
            ConfigCards += [item];
            keys += (string) llGetInventoryKey(item);
        }
    }
    return keys;
}

s_config_state_exit()
{
    llSetTimerEvent(0);
}

integer is_packaged()
{
    list PackageContents_l = ["..MLP Tools - README", ".MENUITEMS.reorient", ".XCITE.example", ".XCITEANIMS.example"];
    integer Count_i = llGetListLength(PackageContents_l);
    integer i;
    for (i=0;i<Count_i;++i)
    {
        if (llGetInventoryType(llList2String(PackageContents_l, i)) == -1) { return FALSE; }
    }
    return TRUE;
}

// ========== STATES =============

// Default state can do any init you need that doesn't require configuration.

default
{
    state_entry() {
        if (is_packaged()) { return; }
        state s_config;
    }
}

// This state is only used to get into s_config, because going from
// s_config to s_config won't redo it's state_entry.  But we might
// not want to redo anything we might have put in default state entry.

state s_reconfig
{
    state_entry() {
        state s_config;
    }
}

// Read card config
// Multiple notecard version - read all cards with the given extension

state s_config
{
    state_entry() {
        config_init();
        ConfigCardKeys = get_cards();
        ConfigCardIndex = 0;

        if (next_card()) {
            llSetTimerEvent(ConfigTimeout);
        } else {
                s_config_state_exit();
            state s_unconfigured;
        }
    }

    dataserver(key query_id, string data) {
        if (query_id == ConfigRequestID) {
            if (data == EOF) {
                if (next_card()) {
                    llSetTimerEvent(ConfigTimeout);
                } else {
                        config_done();
                    s_config_state_exit();
                    state s_active;
                }
            } else {
                    config_parse(data, ConfigCardName, ConfigLineIndex);
                ConfigRequestID = llGetNotecardLine(ConfigCardName, ++ConfigLineIndex);
            }
        }
    }

    timer() {
        say("Dataserver time out");
        s_config_state_exit();
        state s_unconfigured;
    }

    on_rez(integer num) { state s_reconfig; }

    changed(integer change) {
        if (change & CHANGED_OWNER) { llResetScript(); }
        if (change & CHANGED_INVENTORY) { s_config_state_exit(); state s_reconfig; }
    }

    state_exit() {

    }
}


state s_unconfigured
{
    changed(integer change) {
        if (change & CHANGED_INVENTORY) {
            if (get_cards() != ConfigCardKeys) {
                state s_config;
            }
        }
    }
}


state s_active
{

    state_entry() {
        Avnames = [];
        integer ix;
        for (ix = 0; ix < MAX_AVS; ++ix) {
            Avnames += (list) "";
        }
        llSetTimerEvent(Interval);
    }

    link_message(integer from, integer num, string msg, key id) {
        if (num == 0) {
            if (msg == "POSEB") {
                debug(8, "got LM " + (string)num + ": '" + msg + "'");
                setPose((string)id);
                return;
            }
            if (msg == "POSE") {
                debug(8, "got LM " + (string)num + ": '" + msg + "'");
                BallCount = llList2Integer(llCSV2List((string)id), 1);
                return;
            }
            return;
        }

        if (num < -11002 || -11000 < num) {
            return;
        }

        debug(8, "got LM " + (string)num + ": '" + msg + "'");

        list    parms = llParseStringKeepNulls(msg, ["|"], []);
        integer ballnum = (integer)llList2String(parms, 0);
        string  anim = llList2String(parms, 1);

        if (num == -11000) {
            // av hopped on
            // string anim = llList2String(parms, 1);     // anim name parameter, if desired
            debug(1, llKey2Name(id) + ": on ball " + (string)ballnum);
            addAv(llKey2Name(id), ballnum);
            new_anim(id, ballnum, anim);
            setPose(Pose);
        } else if (num == -11001) {
                // av hopped off
                debug(1, llKey2Name(id) + ": off ball " + msg);
            xcite(20014, llKey2Name(id));
            removeAv(ballnum);
        } else if (num == -11002) {
                new_anim(id, ballnum, anim);
        }
    }

    timer() {

        // tickle each av in the pose
        integer bix;                // ball index
        integer cix;                // config index
        string  av;

        for (bix = 0; bix < BallCount; ++bix) {
            av = llList2String(Avnames, bix);
            if (av != "") {
                // llSay(0, Pose + "|" + (string)bix + ":" + llList2CSV(PoseConfigs));
                cix = llListFindList(PoseConfigs, (list)(Pose + "|" + (string)bix));
                if (cix == -1) {
                    // look for default ball on this pose
                    cix = llListFindList(PoseConfigs, (list)(Pose + "|*"));
                    if (cix == -1) {
                        // look for default pose
                        cix = llListFindList(PoseConfigs, (list)"*|*");
                    }
                }

                if (cix != -1) {
                    tickle(av, cix);
                }
            }
        }
    }

    changed(integer change) {
        if (change & CHANGED_INVENTORY) {
            if (get_cards() != ConfigCardKeys) {
                state s_config;
            }
        }
    }
}
