//  [Swimming Critter]

// Moves a thing around the original point.

// Distance in meters that object can move from the start location.
float   DISTANCE_X_FROM_START          = 3.0;
float   DISTANCE_Y_FROM_START          = 3.0;
float   DISTANCE_Z_FROM_START          = 3.0;

float   NUMBER_OF_STEPS_TOTAL          = 40.0;
float   NUMBER_OF_STEPS_START          = 0.0;
float   NUMBER_OF_STEPS_END            = 3.0;

// Time between moving.
float   SLEEP_MAX_SECONDS              = 9.0;
float   SLEEP_MIN_SECONDS              = 1.0;

// If true, don't check to see if we're above the water line.
integer ALLOW_ABOVE_WATER              = FALSE;
// If true, don't rotate to face where we're heading.  (example: jellyfish)
integer IGNORE_FACING                  = FALSE;

/////////////////////////////////////////////////////////////

vector   start_position;
vector   heading;
vector   step;
vector   vPositionCurrent;

vector   headingchange;
float    breathingroom;
float    ground_height;
float    water_height;

float    gfMass;
vector   gvSize;
vector   gvVectorPointing;
float    gfTargetAcquireStrength;
float    gfTargetAcquireDamping;
float    gfMovementStepsMiddle;
float    gfAngleStepX;
float    gfAngleStepY;

vector   vAttitude;

/////////////////////////////////////////////////////////////

float    DAMPING_ADJUSTMENT_DIVISOR    = 20.00;
float    STRENGTH_ADJUSTMENT_DIVISOR   = 2.00;
float    TIMER_OFF                     = 0.00;

//// AXIS_* constants represent the unit vector - 1 unit on the specified axis.
vector   AXIS_FWD                      = <1.00, 0.00, 0.00>;  //  X

/////////////////////////////////////////////////////////////
// Config values of the creature gotten from the "Swimming Critter Settings" script.
integer SIGNAL_NUMBER_OF_CONFIG_VALUES = -1;

integer SIGNAL_DISTANCE_X_FROM_START = 0;
integer SIGNAL_DISTANCE_Y_FROM_START = 1;
integer SIGNAL_DISTANCE_Z_FROM_START = 2;
integer SIGNAL_NUMBER_OF_STEPS_TOTAL = 3;
integer SIGNAL_NUMBER_OF_STEPS_START = 4;
integer SIGNAL_NUMBER_OF_STEPS_END   = 5;
integer SIGNAL_SLEEP_MAX_SECONDS     = 6;
integer SIGNAL_SLEEP_MIN_SECONDS     = 7;
integer SIGNAL_ALLOW_ABOVE_WATER     = 8;
integer SIGNAL_IGNORE_FACING         = 9;

integer gNumberOfConfigValuesGotten = 0;
integer gNumberOfConfigValuesComing = 0;

/////////////////////////////////////////////////////////////

// Returns the rotation required to point the specified axis at the specified position.
//  Maintains orientation of object w/respect to roll axis
rotation GetRotationToPointAxisAt(vector vAxis, vector vTarget)
{
    vector vLocalX = llVecNorm(vTarget - vPositionCurrent);
    vector vLocalY = llVecNorm(<-vLocalX.y, vLocalX.x, 0.00>);
    vector vLocalZ = vLocalX % vLocalY;
    return llAxes2Rot(vLocalX, vLocalY, vLocalZ);
} // GetRotationToPointAxisAt

/////////////////////////////////////////////////////////////

PickNextPosition()
{

    // how far from start point will we be?
    headingchange.x = llFrand( DISTANCE_X_FROM_START * 2.0 ) - DISTANCE_X_FROM_START;
    headingchange.y = llFrand( DISTANCE_Y_FROM_START * 2.0 ) - DISTANCE_Y_FROM_START;
    headingchange.z = llFrand( DISTANCE_Z_FROM_START * 2.0 ) - DISTANCE_Z_FROM_START;

    // so where will we be?
    heading = < headingchange.x + start_position.x,
                headingchange.y + start_position.y,
                headingchange.z + start_position.z >;
        
    breathingroom = gvSize.x / 2.0;
              
    // Avoid digging into the ground
    ground_height = llGround(ZERO_VECTOR); // use zero_vector on objects, and pos on attachments
    if ( (heading.z ) < (ground_height + breathingroom))
    {
        heading.z = ground_height + breathingroom;
    }

    if ( !ALLOW_ABOVE_WATER )
    {
        water_height = llWater(heading); // water height is the same across a sim.
        if ( (heading.z) > ( water_height - breathingroom )  )
        {
            heading.z = water_height - breathingroom;
        }
    }

    // where are we now?
    vPositionCurrent = llGetPos();

    // how far will each step be from current position to final position?
    step = < (heading.x - vPositionCurrent.x) / NUMBER_OF_STEPS_TOTAL,
             (heading.y - vPositionCurrent.y) / NUMBER_OF_STEPS_TOTAL,
             (heading.z - vPositionCurrent.z) / NUMBER_OF_STEPS_TOTAL >;

} // PickNextPosition


/////////////////////////////////////////////////////////////

// Be passive until touched by your owner or someone your group.
default
{
    state_entry()
    {
        llOwnerSay("Touch me when I'm in position.");
    }
    
    on_rez( integer p )
    {
        llOwnerSay("Touch me when I'm in position.");
    }
    
    touch_start( integer d )
    {
        if ( llDetectedKey(0) == llGetOwner() || llSameGroup(llDetectedKey(0)) )
        {
            state initializing;
        }
    }    
}

/////////////////////////////////////////////////////////////

state initializing
{
    state_entry()
    {
        gNumberOfConfigValuesGotten = 0;
        gNumberOfConfigValuesComing = -1;
        
        if ( llGetInventoryType("Swimming Critter Settings") != INVENTORY_SCRIPT )
        {
            llOwnerSay ( "Missing the Swimming Critter Settings script.  Cannot continue." );
            state default;
        }
        
        llMessageLinked( LINK_THIS, SIGNAL_NUMBER_OF_CONFIG_VALUES, "", NULL_KEY);
        
        llSetTimerEvent( 30.0 );
    }
    
    link_message(integer sender_num, integer num, string str, key id)
    {   
        if ( str == "" ) return; // Don't want to hear what we send.  Everything we send has an empty string.
        
        if ( num == SIGNAL_NUMBER_OF_CONFIG_VALUES )
        {
            gNumberOfConfigValuesComing = (integer)str;
            integer x;
            for ( x = 0; x < gNumberOfConfigValuesComing; ++x )
            {
                llMessageLinked( LINK_THIS, x, "", NULL_KEY ); 
            }
        }   
        else if ( num == SIGNAL_DISTANCE_X_FROM_START )
        {
            DISTANCE_X_FROM_START = (float)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_DISTANCE_Y_FROM_START )
        {
            DISTANCE_Y_FROM_START = (float)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_DISTANCE_Z_FROM_START )
        {
            DISTANCE_Z_FROM_START = (float)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_NUMBER_OF_STEPS_TOTAL )
        {
            NUMBER_OF_STEPS_TOTAL = (integer)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_NUMBER_OF_STEPS_START )
        {
            NUMBER_OF_STEPS_START = (integer)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_NUMBER_OF_STEPS_END )
        {
            NUMBER_OF_STEPS_END = (integer)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_SLEEP_MAX_SECONDS )
        {
            SLEEP_MAX_SECONDS = (float)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_SLEEP_MIN_SECONDS )
        {
            SLEEP_MIN_SECONDS = (float)(str);
            ++gNumberOfConfigValuesGotten;
        }
        else if ( num == SIGNAL_ALLOW_ABOVE_WATER )
        {
            ALLOW_ABOVE_WATER  = (integer)(str);
            ++gNumberOfConfigValuesGotten;
        }     
        else if ( num == SIGNAL_IGNORE_FACING )
        {
            IGNORE_FACING = (integer)(str);
            ++gNumberOfConfigValuesGotten;
        }          
        
        // know when to stop!
        if ( gNumberOfConfigValuesGotten == gNumberOfConfigValuesComing )
        {
            llSetTimerEvent( TIMER_OFF );
            state active;
        }
    }
    
    timer()
    {
        llOwnerSay("Initialization failed.  Please Touch to try again.");
        state default;
    }
    
}

/////////////////////////////////////////////////////////////

state active
{

    state_entry()
    {
        llOwnerSay( "Active.");
        //   Strictly, these should all be recalculated if either scale or linking changes
        gvSize = llGetScale();
        gfMass = llGetObjectMass(llGetKey());
        gfTargetAcquireStrength = gfMass / STRENGTH_ADJUSTMENT_DIVISOR;
        gfTargetAcquireDamping = gfTargetAcquireStrength / DAMPING_ADJUSTMENT_DIVISOR;
        gvVectorPointing = AXIS_FWD;
        gfMovementStepsMiddle = NUMBER_OF_STEPS_TOTAL - (NUMBER_OF_STEPS_START + NUMBER_OF_STEPS_END);
    
        // Remember the start position
        start_position = llGetPos();

        llSetTimerEvent( 0.2);
    } // state_entry


    on_rez( integer p )
    {
        llSetTimerEvent(TIMER_OFF);
        state default;
    } // on_rez


    timer()
    {
        llSetTimerEvent(TIMER_OFF);

        // pick a destination
        PickNextPosition();
        
        if ( !IGNORE_FACING )
        {
            llRotLookAt(GetRotationToPointAxisAt(gvVectorPointing, heading),
                        gfTargetAcquireStrength, gfTargetAcquireDamping);
        }
        
        llMessageLinked( LINK_SET, 0, "SWIM", NULL_KEY );
        integer x;

        for ( x = (integer)NUMBER_OF_STEPS_START; x < (integer)gfMovementStepsMiddle; ++x )
        {
            vPositionCurrent = llGetPos();
            // move
            llSetPos( vPositionCurrent + step );
        }

        if ( !IGNORE_FACING )
        {
            llStopLookAt();
            
            vAttitude = llRot2Euler(llGetLocalRot());
            gfAngleStepX = vAttitude.x / NUMBER_OF_STEPS_END;
            gfAngleStepY = vAttitude.y / NUMBER_OF_STEPS_END;

            //  Fair out angles at end of movement (during last steps)
            for ( x = (integer)gfMovementStepsMiddle; x < (integer)NUMBER_OF_STEPS_TOTAL; ++x )
            {
                vAttitude.x -= gfAngleStepX;
                vAttitude.y -= gfAngleStepY;
                llSetPrimitiveParams([PRIM_POSITION, llGetPos() + step, PRIM_ROTATION, llEuler2Rot(vAttitude)]);
            }
        }
        else // IGNORE_FACING
        {
            for ( x = (integer)gfMovementStepsMiddle; x < (integer)NUMBER_OF_STEPS_TOTAL; ++x )
            {
                llSetPrimitiveParams([PRIM_POSITION, llGetPos()]);
            }
        }

        llSetPos( heading );
        
        llMessageLinked( LINK_SET, 0, "STOP", NULL_KEY );
        
        llSetTimerEvent(llFrand(SLEEP_MAX_SECONDS - SLEEP_MIN_SECONDS) + SLEEP_MIN_SECONDS);
    } // timer


} // default


//  [Swimming Critter]