// Communication script for use with an 'iserve' communication strategy.
// Author: Jeff Dalton, modified by Stephen Potter and Austin Tate
// Last modified 2-Aug-2010 by SP
// Copyright (C) 2009-2010, AIAI, The University of Edinburgh

// Modified 2 Aug 10 by SP to include alternative flags for CHANGED_REGION_(RE)START
// Modified 7 Jul 10 by SP to add attempt re-register when status = no-server
// Modified 30 Jun 10 by SP for SL/Opensim compatability
// Modified 22 Jun 10 by AT for capitalisation of scripts and notecards
// Modified 11 Jun 10 by SP to be less chatty
//
//

// The chat channel for commands to this script.
integer COMMAND_CHANNEL = 91;

integer TICK = 600; // 600 seconds for tick to wake up (and check Channel ID for now)

// Is this a critical application? Governs how messages are sent to
// its owner.
integer CRITICAL = FALSE;

// Should we wake up if we hear local chat (channel 0)?
// * In some objects, local chat may lead to a wakeup anyway,
//   for instance if it causes a message to be sent.
// * Local chat is in any case a "weak" wakeup that has an effect
//   only if the script is asleep, not if it thinks there is no server.
integer WAKEUP_ON_LOCAL_CHAT = FALSE;

// Link message commands - these need to be tied to the Help script.
// This is just a convention used with this script.   The command is passed
// as the 2nd parameter (the integer) when llMessageLinked is called.

// Tells this script to send a message.
integer SEND_MESSAGE = 1701;

// Tells other scripts we've received a message.
integer RECEIVED_MESSAGE = 1702;

// Tells this script to output some info.
integer OUTPUT_STATUS = 1703;

// Go to sleep after this many get-next timeouts in a row.
integer MAX_RECEIVE_FAILS = 3;

// Names of configurable parameters that are read from a notecard.
string SERVER = "server";
string SEND_SERVLET = "send-servlet";
string MYNAME = "myname";

// Commands to this script.
string REGISTER = "register";
string RESET = "comms reset";
string START = "start";          // valid only in default state
string STATUS = "status";
string TRACE_ON = "trace on";
string TRACE_OFF = "trace off";
string WAKEUP = "wakeup";

// Status values
string STATUS_STARTUP = "startup";
string STATUS_WORKING = "working";
string STATUS_ASLEEP = "asleep";
string STATUS_NO_SERVER = "no server";

string status = "";

// A URL of a resource that contains one line telling us where the server is.
// We set this when we read the config notecard.
string service_url = "Not a service_url";

// The *base* URL for requests to the server.
string comm_server_url = "Not a request URL";

// The servlet that should handle messages we send
string send_servlet = "";

// URL for our send requests directly to the comm server.
// This is made from the comm_server_url, the send_servlet, and other things.
string http_send_request_url = "Not a send URL";

// Mesages waiting to be sent
list waiting = [];

// URL for our receive 'get-next' requests directly to the comm server.
string http_receive_request_url = "Not a get-next URL";

string receive_ack = "This is my first get-next request";
integer receive_fail_count = 0;
string last_id_received = "";

// URL for 'register' requests directly to the comm server
string http_register_request_url = "Not a register URL";

// Ids for outstanding HTTP requests.  NULL_KEY when no request is active.
// Only one request of each type is ever active at a time.
key http_request_id = NULL_KEY;         // for configuring and similar
key http_send_request_id = NULL_KEY;
key http_receive_request_id = NULL_KEY;

// Channel for XML-RPC
key xml_rpc_channel = NULL_KEY;

// For reading the config note card:
string CONFIG_CARD_NAME = "I-X Comms Setup";
integer card_line = 0;
key card_query_id = NULL_KEY;

// 'traceSay' messages are written to DEBUG_CHANNEL to indicate progress or info,
// rather than bugs or problems, and are turned off during normal use.
// 'debugSay' is for messages that should always appear.

integer TRACE = FALSE; // TRUE or FALSE

integer timestamp_debug_output = FALSE; // becomes TRUE in state 'working'.

key owner_id = NULL_KEY;

traceSay(string message) {
    if (TRACE) llSay(DEBUG_CHANNEL, message);
}

debugSay(string message) {
    if (TRACE) {
        if (timestamp_debug_output) {
            llSay(DEBUG_CHANNEL, llGetTimestamp());
        }
        llSay(DEBUG_CHANNEL, message);
    }
    else {
        if(CRITICAL){
            llInstantMessage(owner_id, message);
        }
        else {
            llOwnerSay(message);
        }
    }
}

process_config_line(string text) {
    traceSay(text);
    text = llStringTrim(text, STRING_TRIM);
    if (text == "") { // blank line
        return;
    }
    if (llGetSubString(text, 0, 0) == "#") {  // a comment
        return;
    }
    if (llGetSubString(text, 0, 1) == "//") { // a comment
        return;
    }
    string name = left(text, "=");
    string value = right(text, "=");
    traceSay("Setting " + name + " = " + value);
    if (name == SERVER) {
        service_url = value;
    }
    else if (name == SEND_SERVLET) {
        send_servlet = value;
    }
    else if (name == MYNAME) {
        // Not currently used.  /\/
    }
    else {
        debugSay("Invalid config line: " + text);
    }
}

set_status(string s) {
    traceSay("Setting status from " + status + " to " + s);
    status = s;
}

say_status(key id) {
    integer n_waiting = (waiting != []);
    string message = "Status = " + status;
    if (n_waiting > 0) {
        message += "\n" + (string)n_waiting + " messages waiting to be sent.";
    }
    if(CRITICAL) {
        llInstantMessage(id, message);
    }
    else {
        llOwnerSay(message);
    }
        
}

set_asleep() {
    set_status(STATUS_ASLEEP);
}

set_no_server() {
    set_status(STATUS_NO_SERVER);
}

ensure_awake(string reason) {
    if (status != STATUS_WORKING) {
        traceSay("Will wake up because: " + reason);
        wake_up(reason);
    }
}

wake_up(string reason) {
    traceSay("Waking up while status = " + status);
    set_status(STATUS_WORKING);
    // Restart sending
    if (http_send_request_id == NULL_KEY) {
        send_if_next_message();
    }
    else {
        traceSay("A send request was still outstanding");
    }
    // Restart receiving
    if (http_receive_request_id == NULL_KEY) {
        receive_fail_count = 0;
        send_next_receive_request();
    }
    else {
        traceSay("A receive request was still outstanding");
    }
}

key send_register_request() {
    // traceSay("Registering at " + comm_server_url);
    string body = xml_rpc_channel;
    return llHTTPRequest(http_register_request_url, [HTTP_METHOD, "POST"], body);
}

send_if_next_message() {
    // Complain if we're trying to send before we've finished with the
    // previous request.
    if (http_send_request_id != NULL_KEY) {
        debugSay("Nested send attempts");
    }
    // Else if there's another message in the queue, try to send it.
    else if (waiting != []) {
        http_send_request_id = send_next_message();
    }
}

key send_next_message() {
    // Sends the first message in 'waiting'.
    key id = NULL_KEY;
    while (id == NULL_KEY) {
        integer len = (waiting != []);
        if (len > 1) {
            traceSay((string)len + " waiting");
        }
        string body = llList2String(waiting, 0);
        traceSay("Sending: " + body);
        id = llHTTPRequest(http_send_request_url, [HTTP_METHOD, "POST"], body);
        if (id == NULL_KEY) {
            debugSay("Send throttled; " + (string)len + " waiting");
            llSleep(5.0);
        }
    }
    return id;
}

remove_sent_message() {
    // Deletes the first message 'waiting'.
    waiting = llDeleteSubList(waiting, 0, 0);
}

send_next_receive_request() {
    http_receive_request_id = send_receive_request(receive_ack);
}

key send_receive_request(string body) {
    while(1) {
        traceSay("Get-next request: " + body);
        key id = llHTTPRequest(http_receive_request_url, [HTTP_METHOD, "POST"], body);
        if (id != NULL_KEY) {
            return id;
        }
        debugSay("get-next throttled");
        llSleep(5.0);
    }
    return NULL_KEY; // Never gets here.
}

integer is_avatar_id(key id) {
    return llGetOwnerKey(id) == id;
}

// If the separator does not appear in the source string, then this version
// of 'left' returns "", and this 'right' returns the whole strong.

string left(string source, string separator) {
    integer i = llSubStringIndex(source, separator);
    if (~i)
        return llDeleteSubString(source, i, -1);
    else
        return "";
}

string right(string source, string separator) {
    integer i = llSubStringIndex(source, separator);
    if (~i)
        return llDeleteSubString(source, 0, i + llStringLength(separator) - 1);
    else
        return source;
}

list breakup(string message) {
    return llParseStringKeepNulls(message, ["^"], []);
}

string reform(list broken_message) {
    return llDumpList2String(broken_message, "^");
}

default
{
    state_entry() {
        owner_id = llGetOwner(); // for debug output
        llListen(COMMAND_CHANNEL, "", NULL_KEY, START);
        set_status(STATUS_STARTUP);
        debugSay("To start I-X communications, type /" +
                    (string)COMMAND_CHANNEL + " " + START);
    }
    
    on_rez(integer start_param) {
        // Every time we're rezzed, reset the script.
        // This ensures that all local variables are set afresh.
        // We have to do this in every state that might be the current
        // state when the object is copied.
        llResetScript();
    }
    
    listen(integer channel, string name, key id, string message )
    {
        // We've arranged, via the llListen above, to get here only if there's
        // a "start" on the COMMAND_CHANNEL. 
        set_status(STATUS_STARTUP);
        state configure;
    }
    
    link_message(integer sender_num, integer num, string message, key id)
    {   
        if (num == OUTPUT_STATUS) {
            //llSay(0, "Venue: "+"Command channel: "+"Not currently registered");
            llDialog(id,"I-X Comms Status\nNot currently registered.\nClick below to start registration...",["start"],COMMAND_CHANNEL);
        }
    }
    
}

state configure
{
    
    state_entry()
    {
        debugSay("Reading card " + CONFIG_CARD_NAME);
        card_line = 0;
        card_query_id = llGetNotecardLine(CONFIG_CARD_NAME, card_line);
    }
    
    on_rez(integer start_param) {
        llResetScript();
    }
    
    dataserver(key query_id, string text) {
        if (query_id == card_query_id) {
            if (text == EOF) {
                // We've read everything in the card.
                state get_server_url;
            }
            else {
                // Process the line.
                process_config_line(text);
                // Request the next line.
                ++card_line;
                card_query_id = llGetNotecardLine(CONFIG_CARD_NAME, card_line);
            }
        }
    }

}

state get_server_url
{
    state_entry()
    {
        // Check whether we have a valid service_url.
        if (service_url == "Not a service_url") {
            debugSay("No server was specified in note card " + CONFIG_CARD_NAME);
            llResetScript();
        }
        else if (llSubStringIndex(service_url, " ") >= 0) {
            debugSay("The server url in " + CONFIG_CARD_NAME + " contains spaces.");
            llResetScript();
        }
        debugSay("Asking for server URL.");
        http_request_id = llHTTPRequest(service_url, [HTTP_METHOD, "GET"], "");
        if (http_request_id == NULL_KEY) {
            debugSay("Grrr.  Wakeup script throttled while asking for server URL.");
            // llSay(0, "Sorry, but I can't start now. Try again in 30 seconds.");
            llResetScript();
        }
    }
    
    on_rez(integer start_param) {
        llResetScript();
    }

    http_response(key request_id, integer status, list metadata, string body)
    {
        if (request_id == http_request_id)
        {
            http_request_id = NULL_KEY; // we're through with that id
            if (status == 200) {
                string server_url = llStringTrim(body, STRING_TRIM);
                debugSay("Server URL = " + server_url);
                comm_server_url = server_url;
                state get_xml_rpc_channel;
            }
            else {
                debugSay("Could not get server URL from " + service_url + " status:" +
                         (string)status + ": " + body);
                llResetScript();
            }
        }
    }
    
    state_exit()
    {
        // Set values that depend on the server URL + config info.
        
        http_send_request_url = comm_server_url + "/ipc/sl/" + send_servlet;
        debugSay("Send request url = " + http_send_request_url);
        
        http_receive_request_url = comm_server_url + "/ipc/sl/get-next";
        debugSay("Receive request url = " + http_receive_request_url);
        
        http_register_request_url = comm_server_url + "/ipc/sl/register";
        
    }
    
}

state get_xml_rpc_channel
{
    state_entry()
    {
        // Open a remote data channel for XML-RPC
        llOpenRemoteDataChannel();
    }
    
    on_rez(integer start_param) {
        llResetScript();
    }
    
    remote_data(integer type, key channel, key uid, string from, integer
                intValue, string strValue)
    {
        if (type == REMOTE_DATA_CHANNEL)
        {
            // We've received a channel, presumably the one we asked for.
            xml_rpc_channel = channel;
            // Register the channel with the comm server.
            state register;
        }
        else if (type == REMOTE_DATA_REQUEST)
        {
            // We've received an XML-RPC message.
            debugSay("Received RPC from " + from + ": " + 
                        (string)intValue + ", " + strValue);
        }
    }

}

state register
{
    state_entry()
    {
        debugSay("Registering at " + comm_server_url);
        http_request_id = send_register_request();
        if (http_request_id == NULL_KEY)
        {
            debugSay("Grrr.  Send script throttled while trying to register.");
            // llSay(0, "Sorry, but I can't send the registration request now.  " +
            //          "Try again in 30 seconds.");
            llResetScript();
        }
    }
    
    on_rez(integer start_param) {
        llResetScript();
    }

    
    http_response(key request_id, integer status, list metadata, string body)
    {
        if (request_id == http_request_id)
        {
            http_request_id = NULL_KEY; // we're through with that id
            if (status == 200) {
                // We've successfully sent a registration message.
                if (body == "OK") {
                    debugSay("Registered");
                    state working;
                }
                else {
                    debugSay("Response to registration = " + body);
                    llResetScript();
                }
            }
            else if (status == 503) {
                 // 503 = service unavailable.  This is what SL normally gives
                 // us when the server isn't running.
                 debugSay("Registration attempt got service unavailable");
                 llResetScript();
            }
            else {
                debugSay("Registration status " + (string)status + 
                         "; Received: " + body);
                llResetScript();
            }
        }
    }
    
}

state working
{
    // Note that if we do a state change, all pending events are deleted,
    // including queued link_messages, so we stay in this state even
    // when things go wrong, hoping that they will soon be fixed.
    
    state_entry()
    {
        timestamp_debug_output = TRUE;
        
        debugSay("Comm script's started working");
        
        http_request_id = NULL_KEY; // just to make sure
                
        // Set how frequently we will check to see whether we still have the
        // same XML-PRC channel.
        llSetTimerEvent(TICK);  // TICK was 10 * 60
        
        // We may want to wake up if anything is said in our vicinity.
        if (WAKEUP_ON_LOCAL_CHAT) {
            llListen(0, "", NULL_KEY, "" );
        }
        
        // We also want to receive commands.
        llListen(COMMAND_CHANNEL, "", NULL_KEY, "");
        
        set_status(STATUS_WORKING);
        
        // Start sending if there's anything to send.
        send_if_next_message();
        
        // Start receiving
        receive_fail_count = 0;
        send_next_receive_request();
    }
    
    on_rez(integer start_param) {
        llResetScript();
    }
       
    listen(integer channel, string name, key id, string message )
    {
        // traceSay("Listen: " + (string)channel + ": " + message);
        if (channel == 0) {
            // Wake up if asleep when we hear any public local chat.
            // N.B. We'll hear chat only if we called llListen(0, ...)
            // in state_entry().
            if (status == STATUS_ASLEEP) {
                // Only if ALSEEP, not if NO_SERVER.
                ensure_awake("chat");
            }
        }
        else if (channel == COMMAND_CHANNEL) {
            if (message == WAKEUP) {
                wake_up("wakeup command");
            }
            else if (message == RESET) {
                llResetScript();
            }
            else if (message == REGISTER) {
                // Let's make sure we have an up-to-date XML-RPC channel too.
                // Forgetting the channel, then asking for one, will cause
                // the remote_data event to send a new registration.
                xml_rpc_channel = NULL_KEY;
                llOpenRemoteDataChannel();
            }
            else if (message == STATUS) {
                if (is_avatar_id(id)) {
                    say_status(id);
                }
            }
            else if (message == TRACE_ON) {
                TRACE = TRUE;
                traceSay("Ok.  Trace is on.");
            }
            else if (message == TRACE_OFF) {
                traceSay("Turning trace off.");
                TRACE = FALSE;
            }
            else {
                debugSay("Invalid command to wakeup script: " + message);
            }
        }
    }

    remote_data(integer type, key channel, key uid, string from, integer
                intValue, string strValue)
    {
        if (type == REMOTE_DATA_REQUEST)
        {
            // We've received an XML-RPC message.
            traceSay("Received RPC from " + from + ": " +
                        (string)intValue + ", " + strValue);
            if (channel != xml_rpc_channel)
                debugSay("channel " + (string)channel + ", expected " +
                            (string)xml_rpc_channel);
            traceSay("Comm server said time to wake up!");
            wake_up("comm server sent wakeup");
            llRemoteDataReply(channel, NULL_KEY, "OK", 0);
        }
        else if (type == REMOTE_DATA_CHANNEL)
        {
            // We've received a channel, presumably the one we asked for.
            traceSay("Received data channel " + (string)channel);
            // See if we have a new channel.   The 1st time we request a channel
            // it should seem new, because xml_rpc_channel is initially NULL_KEY.
            if (channel == xml_rpc_channel && status != STATUS_NO_SERVER) {
                // The channel's the same as before, and we still have a server
                traceSay("Still have server and channel " + (string)xml_rpc_channel);
            }
            else {
                // We have a new channel, or the server is gone.
                xml_rpc_channel = channel;
                // Register the channel with the comm server.
                key id = NULL_KEY;
                while (id == NULL_KEY) {
                    id = send_register_request();
                    if (id == NULL_KEY) {
                        debugSay("Registration throttled");
                        llSleep(5.0);
                    }
                }
                http_request_id = id;
            }
        }
    }
    
    timer()
    {
        traceSay("Tick");
        // llSetTimerEvent(0.0); // disable timer

        // Request a remote data channel for XML-RPC. To allow comparison to previous CID
        llOpenRemoteDataChannel();
    }
    
    changed (integer change)
    {
        // CHANGED_REGION_START doesn't seem to exist yet.  /\/
        // Called CHANGED_REGION_START on Second Life but .._RESTART on Opensim
        //if (change & CHANGED_REGION_START)
        //if (change & CHANGED_REGION_RESTART)
        // CHANGED_REGION_START   = 0x400 = 1024 = Second Life
        // CHANGED_REGION_RESTART = 0x100 = 256 = Opensim
        if((change & 0x400) || (change & 0x100))
        {
            // The region has restarted.
            // Request a remote data channel for XML-RPC.
            llOpenRemoteDataChannel();
        }
    }
    
    link_message(integer sender_num, integer num, string message, key id)
    {
        if (num == SEND_MESSAGE) {
            // We've been given a message to send.
            traceSay("New message to send: " + message);
            // Put it in the queue.
            waiting += message;
            // If we're not waiting for a response to an earlier send,
            // we can try to send it now.
            if (http_send_request_id == NULL_KEY) {
                http_send_request_id = send_next_message();
            }
            else {
                traceSay("Can't send yet because there's an active request");
                integer len = (waiting != []);
                traceSay((string)len + " waiting");
            }
        }
        
        if (num == OUTPUT_STATUS) {
            integer n_waiting = (waiting != []);
            string stmessage = "Status = " + status;
            if (n_waiting > 0) {
                stmessage += "\n" + (string)n_waiting + " messages waiting to be sent.";
            }
            string TRACE_CMD = TRACE_ON;
            if(TRACE) TRACE_CMD = TRACE_OFF;
            llDialog(id,"I-X Comms Status\nRegistration details:\nServer = "+comm_server_url+"\n"+stmessage,[WAKEUP,RESET,TRACE_CMD],COMMAND_CHANNEL); 
        }
    }
    
    http_response(key request_id, integer status, list metadata, string body)
    {
        if (request_id == http_send_request_id)
        {
            http_send_request_id = NULL_KEY; // we're through with that id
            if (status == 200) {
                // We've successfully sent a message.
                if (body != "OK")
                    debugSay("Response to send = " + body);
                // Remove the sent message from the queue.
                remove_sent_message();
                // If there's another message in the queue, try to send it.
                send_if_next_message();
                // Ensure we're awake, since we know the server is responding.
                // N.B. We have to do this *after* the send_if_next_message call.
                // Ensure_awake might send a message, and then send_if_next_message
                // would complain about a nested send attempt.  Ensure_awake, OTOH,
                // does not mind if a send is already in progress.
                ensure_awake("successful send");
            }
            else if (status == 400) {
                // 400 = bad request.  We get this if there's a syntax error
                // in what we sent, including in the contents.
                debugSay("Send received 400: " + body);
                // Remove the message.  (We assume sending it again won't work.)
                remove_sent_message();
                // If there's another message in the queue, try to send it.
                send_if_next_message();
            }
            // In the remaining error cases, we won't keep the sending cycle going
            // just because we still have messages waiting to be sent.   So we'll
            // wait for something new to send or a WAKEUP control message.
            else if (status == 503) {
                 // 503 = service unavailable.  This is what SL normally gives
                 // us when the server isn't running.
                 debugSay("Send script got service unavailable");
                 set_no_server();
            }
            else {
                debugSay("Send req status " + (string)status + 
                         "; Received: " + body);
                set_no_server();
            }
        }
        else if (request_id == http_receive_request_id)
        {
            http_receive_request_id = NULL_KEY; // we're through with that id
            if (status == 200) {
                // We've received something.
                ensure_awake("received a message");
                receive_fail_count = 0;
                // See if it's something new.
                traceSay("get-next received: " + body);
                string sequence_id = left(body, "^");
                if (sequence_id) {
                    if (sequence_id != last_id_received) {
                        // It is new, so pass it on.
                        body = right(body, "^");
                        last_id_received = sequence_id;
                        llMessageLinked(LINK_THIS, RECEIVED_MESSAGE, body, "");
                    }
                    else {
                        traceSay("Received " + sequence_id + " again");
                    }
                    receive_ack = "Received " + sequence_id;
                }
                else {
                    //\/: No sequence_id, so we assume the message is new.
                    traceSay("Assuming received message is new.");
                    llMessageLinked(LINK_THIS, RECEIVED_MESSAGE, body, "");
                    receive_ack = "Thank you for: " + body;
                }
                send_next_receive_request();
            }
            else if (status == 499) {
                // Usually this is a timeout.
                traceSay("Get-next status " + (string)status + "; Received: " + body);
                receive_fail_count += 1;
                traceSay((string)receive_fail_count + " failures in a row.");
                if (receive_fail_count >= MAX_RECEIVE_FAILS) {
                    set_asleep();
                }
                else {
                    send_next_receive_request();
                }
            }
            else if (status == 503) {
                 // 503 = service unavailable.  This is what SL normally gives
                 // us when the server isn't running.
                 debugSay("Get-next got service unavailable");
                 set_no_server();
            }
            else if (status == 502) {
                // 502 = bad gateway, meaning the SL server got an invalid response
                // from our server.  We get this if there's a zero-length response,
                // which happens if the server goes down while we have a request
                // outstanding.
                debugSay("Get-next: invalid upstream response");
                set_no_server();
            }
            else {
                debugSay("Get-next status " + (string)status + "; Received: " + body);
                set_no_server();
            }
        }
        else if (request_id == http_request_id) // registration
        {
            http_request_id = NULL_KEY; // we're through with that id
            if (status == 200) {
                // We've successfully registered.
                if (body == "OK") {
                    debugSay("Registered");
                }
                else {
                    debugSay("Response to registration = " + body);
                }
                ensure_awake("successful registration");
            }
            else if (status == 503) {
                 // 503 = service unavailable.  This is what SL normally gives
                 // us when the server isn't running.
                 debugSay("Registration attempt got service unavailable");
                 set_no_server();
            }
            else {
                debugSay("Registration status " + (string)status + 
                         "; Received: " + body);
                set_no_server();
            }
        }
    }
    
}
