 /////////////// CONSTANTS ///////////////////
 // XyText Message Map.
 integer DISPLAY_STRING      = 201000;
 integer DISPLAY_EXTENDED    = 201001;
 integer REMAP_INDICES       = 201002;
 integer RESET_INDICES       = 201003;
 integer SET_CELL_INFO       = 201004;
 
 // This is an extended character escape sequence.
 string  ESCAPE_SEQUENCE = "\\e";
 
 // This is used to get an index for the extended character.
 string  EXTENDED_INDEX  = "123456789abcdef";
 
 // Face numbers.
 integer LEFT_FACE       = 4;
 integer MIDDLE_FACE     = 0;
 integer RIGHT_FACE      = 2;
 
 // This is a list of textures for all 2-character combinations.
key     TRANSPARENT     = "b5fc01f6-2b4a-4e42-855d-6872ea360aae";

list    CHARACTER_GRID  = ["5051e617-e7ed-4239-823e-0a37961de44d","4eea18ba-d099-4ae3-bddf-342ba245c28c","1d1fe3ae-b49d-4286-b920-f90754878faa","1e8a804b-8f69-42e2-8ecd-2b54ab57f06d","769da83a-296b-4866-b7ae-5fac46ccdefa","88667afc-3046-475f-8fc8-f311cd161677","70a1a2a8-683d-496b-8fc8-995ac01ab155","ef50a6da-3776-4558-bdc9-b2afa619701f","b8b32073-42f0-487b-bfeb-8b51a8f4addf","42919db0-ca2c-43a6-8192-18326efeefc2","774e17b0-6d21-48c4-a702-919870aae20a","68725957-93b4-4b50-8947-348a8e9f8cfa","2cb35ff9-80f6-4584-b80c-934b18ced081","9cdc8bf0-5c31-465a-998a-2b17970bbdf1","0063f252-6f46-41c1-be18-caa9634af070","f093402c-39ef-44b4-a062-7f7956f2f19e","55728875-66b7-4eea-8a4a-9f697738fad8","55febbe7-4904-41fc-9807-09cda12e72ae","f6f5181e-4ca0-4fec-b29a-7fa639745deb","0d9eea9d-0043-486f-ba46-c36576798006","4b7b95a6-a24e-4076-ad10-436760c99c98","29e3ecd4-c607-403d-817a-a1eb6b2a7873","50469d2e-b6c2-4c50-9835-c63cebb7f71c","c27a83e7-1b69-47a1-9a9e-c26f6640662a","90a06661-e6da-4d3a-b64c-9bf3c5f1203e","cc439e14-5003-4868-8e87-392214d56626","08af3bc7-3e5e-4a66-98e2-3c84c00b99d5","89891e48-ef70-4f84-9cc4-9bb5788396d5","be53061d-08f8-4538-85b8-57bdf6f42f50","049f3024-3dad-4f91-87db-21e82ada00a2","212156f1-b8ad-47fa-a143-ea5548820112","d2c63c71-3b7d-4a6e-8223-bef98f5ce97c","8e4a6f7d-e92f-451f-b1d6-2e8cab186ce2","9b2711d5-f0a6-4ce1-b419-e07dbd3568bf","3acda0f1-0420-4386-aa31-3ac1ecadfdcc","db38cc84-db09-464a-aab6-ca93921c9f47","50414d73-eac5-44ca-a816-5049f5e87869","8153fe4a-0c6b-4bdc-bb85-2e2c091bfab4","4b9421e4-ef68-4aa9-8dc5-9024786d315b","46a6854e-98ff-48f2-91f9-5f0dfdb11ef9","e9589e63-f2e1-40da-ad09-27c9b5a9b88a","1b1eca43-d41a-4af1-b345-cda7f6a62d8a","f0953302-adf2-499d-b0b5-df6cb9844e7c","ca39e990-84dc-4860-9041-d80917a5df8c","8d615593-0c91-413c-90f0-16cabbffda22","439314d1-d535-4f12-9a99-3ef4f8c6ad53","b7d33ba6-ae2d-4da8-8b7b-fa547569d44c","4afed66b-4820-4f07-9805-df1c8be0b6e1","7ca90368-3d7d-4678-9544-e5389a3e922a","20b43f4f-c5ce-42f6-a4d7-79b24678110b","fdeb9c73-e872-4186-920a-ba3bf5c01e50","ef753bb0-7baa-43ac-a680-354b51d0d199","565e97bb-723b-4b0b-811e-83737903a9b7","05c37788-79f4-4a92-ab48-8e6cfd1581a7","15d7dbca-b0d3-4b9a-ae9e-5f9cdda1a6d5","184f2baa-bc5e-4d06-a5ad-08279dd6951a","4b5f4927-8607-43ce-8a02-b848662fd5e7","9ac1cfb9-8142-4443-bfbb-49c5abe8f0e1","748e44d1-c28c-4af3-b27a-30a7b0913955","96d9e77b-fb49-4356-a756-0df4d4200334","2987060d-3595-407f-9b97-e8ceb42f5f23","88a2cb57-857a-4933-ac02-08fdd429b4b2","d5b9e529-81aa-4855-b446-987f16b0b30f","08bd3029-e111-4387-af51-e53f2a943051","caf2b3e1-9962-4100-84b7-239c1be409a2","6b4d6103-1242-4705-a9da-7f895fe75a95"];
 
 ///////////// END CONSTANTS ////////////////
 
 ///////////// GLOBAL VARIABLES ///////////////
 // All displayable characters.  Default to ASCII order.
 string gCharIndex;
 // This is the channel to listen on while acting
 // as a cell in a larger display.
 integer gCellChannel      = -1;
 // This is the starting character position in the cell channel message
 // to render.
 integer gCellCharPosition = 0;
 /////////// END GLOBAL VARIABLES ////////////
 
 ResetCharIndex() {
     gCharIndex  = " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`";
     // \" <-- Fixes LSL syntax highlighting bug.
     gCharIndex += "abcdefghijklmnopqrstuvwxyz{|}~";
     gCharIndex += "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n";
 }
 
 vector GetGridPos(integer index1, integer index2) {
     // There are two ways to use the lookup table...
     integer Col;
     integer Row;
     if (index1 >= index2) {
         // In this case, the row is the index of the first character:
         Row = index1;
         // And the col is the index of the second character (x2)
         Col = index2 * 2;
     }
     else { // Index1 < Index2
         // In this case, the row is the index of the second character:
         Row = index2;
         // And the col is the index of the first character, x2, offset by 1.
         Col = index1 * 2 + 1;
     }
     return < Col, Row, 0>;
 }
 
 string GetGridTexture(vector grid_pos) {
     // Calculate the texture in the grid to use.
     integer GridCol = llRound(grid_pos.x) / 20;
     integer GridRow = llRound(grid_pos.y) / 10;
 
     // Lookup the texture.
     key Texture = llList2Key(CHARACTER_GRID, GridRow * (GridRow + 1) / 2 + GridCol);
     return Texture;
 }
 
 vector GetGridOffset(vector grid_pos) {
     // Zoom in on the texture showing our character pair.
     integer Col = llRound(grid_pos.x) % 20;
     integer Row = llRound(grid_pos.y) % 10;
 
     // Return the offset in the texture.
     return <-0.45 + 0.05 * Col, 0.45 - 0.1 * Row, 0.0>;
 }
 
 ShowChars(vector grid_pos1, vector grid_pos2, vector grid_pos3) {
     // Set the primitive textures directly.
     llSetPrimitiveParams( [
         PRIM_TEXTURE, LEFT_FACE,   GetGridTexture(grid_pos1), <0.1, 0.1, 0>, GetGridOffset(grid_pos1), PI_BY_TWO,
         PRIM_TEXTURE, MIDDLE_FACE, GetGridTexture(grid_pos2), <0.1, 0.1, 0>, GetGridOffset(grid_pos2), 0.0,
         PRIM_TEXTURE, RIGHT_FACE,  GetGridTexture(grid_pos3), <0.1, 0.1, 0>, GetGridOffset(grid_pos3), -PI_BY_TWO
         ]);
 }
 
 RenderString(string str) {
     // Get the grid positions for each pair of characters.
     vector GridPos1 = GetGridPos( llSubStringIndex(gCharIndex, llGetSubString(str, 0, 0)),
                                   llSubStringIndex(gCharIndex, llGetSubString(str, 1, 1)) );
     vector GridPos2 = GetGridPos( llSubStringIndex(gCharIndex, llGetSubString(str, 2, 2)),
                                   llSubStringIndex(gCharIndex, llGetSubString(str, 3, 3)) );
     vector GridPos3 = GetGridPos( llSubStringIndex(gCharIndex, llGetSubString(str, 4, 4)),
                                   llSubStringIndex(gCharIndex, llGetSubString(str, 5, 5)) );
 
     // Use these grid positions to display the correct textures/offsets.
     ShowChars(GridPos1, GridPos2, GridPos3);
 }
 
 RenderExtended(string str) {
     // Look for escape sequences.
     list Parsed       = llParseString2List(str, [], [ESCAPE_SEQUENCE]);
     integer ParsedLen = llGetListLength(Parsed);
 
     // Create a list of index values to work with.
     list Indices;
     // We start with room for 6 indices.
     integer IndicesLeft = 6;
 
     integer i;
     string Token;
     integer Clipped;
     integer LastWasEscapeSequence = FALSE;
     // Work from left to right.
     for (i = 0; i < ParsedLen && IndicesLeft > 0; i++) {
         Token = llList2String(Parsed, i);
 
         // If this is an escape sequence, just set the flag and move on.
         if (Token == ESCAPE_SEQUENCE) {
             LastWasEscapeSequence = TRUE;
         }
         else { // Token != ESCAPE_SEQUENCE
             // Otherwise this is a normal token.  Check its length.
             Clipped = FALSE;
             integer TokenLength = llStringLength(Token);
             // Clip if necessary.
             if (TokenLength > IndicesLeft) {
                 Token = llGetSubString(Token, 0, IndicesLeft - 1);
                 TokenLength = llStringLength(Token);
                 IndicesLeft = 0;
                 Clipped = TRUE;
             }
             else
                 IndicesLeft -= TokenLength;
 
             // Was the previous token an escape sequence?
             if (LastWasEscapeSequence) {
                 // Yes, the first character is an escape character, the rest are normal.
 
                 // This is the extended character.
                 Indices += [llSubStringIndex(EXTENDED_INDEX, llGetSubString(Token, 0, 0)) + 95];
 
                 // These are the normal characters.
                 integer j;
                 for (j = 1; j < TokenLength; j++)
                     Indices += [llSubStringIndex(gCharIndex, llGetSubString(Token, j, j))];
             }
             else { // Normal string.
                 // Just add the characters normally.
                 integer j;
                 for (j = 0; j < TokenLength; j++)
                     Indices += [llSubStringIndex(gCharIndex, llGetSubString(Token, j, j))];
             }
 
             // Unset this flag, since this was not an escape sequence.
             LastWasEscapeSequence = FALSE;
         }
     }
 
     // Use the indices to create grid positions.
     vector GridPos1 = GetGridPos( llList2Integer(Indices, 0), llList2Integer(Indices, 1) );
     vector GridPos2 = GetGridPos( llList2Integer(Indices, 2), llList2Integer(Indices, 3) );
     vector GridPos3 = GetGridPos( llList2Integer(Indices, 4), llList2Integer(Indices, 5) );
 
     // Use these grid positions to display the correct textures/offsets.
     ShowChars(GridPos1, GridPos2, GridPos3);
 }
 
 integer ConvertIndex(integer index) {
     // This converts from an ASCII based index to our indexing scheme.
     if (index >= 32) // ' ' or higher
         index -= 32;
     else { // index < 32
         // Quick bounds check.
         if (index > 15)
             index = 15;
 
         index += 94; // extended characters
     }
 
     return index;
 }
 
 default {
     state_entry() {
         // Initialize the character index.
         ResetCharIndex();
 
         //llSay(0, "Free Memory: " + (string) llGetFreeMemory());
     }
 
     link_message(integer sender, integer channel, string data, key id) {
         if (channel == DISPLAY_STRING) {
             RenderString(data);
             return;
         }
         if (channel == DISPLAY_EXTENDED) {
             RenderExtended(data);
             return;
         }
         if (channel == gCellChannel) {
             // Extract the characters we are interested in, and use those to render.
             RenderString( llGetSubString(data, gCellCharPosition, gCellCharPosition + 5) );
             return;
         }
         if (channel == REMAP_INDICES) {
             // Parse the message, splitting it up into index values.
             list Parsed = llCSV2List(data);
             integer i;
             // Go through the list and swap each pair of indices.
             for (i = 0; i < llGetListLength(Parsed); i += 2) {
                 integer Index1 = ConvertIndex( llList2Integer(Parsed, i) );
                 integer Index2 = ConvertIndex( llList2Integer(Parsed, i + 1) );
 
                 // Swap these index values.
                 string Value1 = llGetSubString(gCharIndex, Index1, Index1);
                 string Value2 = llGetSubString(gCharIndex, Index2, Index2);
 
                 gCharIndex = llDeleteSubString(gCharIndex, Index1, Index1);
                 gCharIndex = llInsertString(gCharIndex, Index1, Value2);
 
                 gCharIndex = llDeleteSubString(gCharIndex, Index2, Index2);
                 gCharIndex = llInsertString(gCharIndex, Index2, Value1);
             }
             return;
         }
         if (channel == RESET_INDICES) {
             // Restore the character index back to default settings.
             ResetCharIndex();
             return;
         }
         if (channel == SET_CELL_INFO) {
             // Change the channel we listen to for cell commands, and the
             // starting character position to extract from.
             list Parsed = llCSV2List(data);
             gCellChannel        = (integer) llList2String(Parsed, 0);;
             gCellCharPosition   = (integer) llList2String(Parsed, 1);
             return;
         }
     }
 }