// Deluxe Visitor Log - keeps track of visitors to the region, storing the information in a notecard to preserve
//                      the data between script resets, region restarts and the like. When touched by the owner,
//                      produces a notecard containing the visitor list sorted by number of visits, alphabetically
//                      by visitor name, and by date of first visit. (If anyone other than the owner touches it,
//                      they are simply told the number of visitors the region has had.)
//
// After putting this script into a prim, open the menu by touching the prim and select
// "Reset" to enable the tracing of logging start date and most recent visit date.
//
// Requires the following OSSL functions: osMakeNotecard(), osGetAgents(), osAvatarName2Key(), osSetDynamicTextureData().
// Make sure they are allowed in OpenSim.ini for the UUID of the avatar owning the object which contains the script.
//
// Version 1.0 by Warin Cascabel (4 July 2009)
// Version 1.1 by Warin Cascabel (6 July 2009) - set object description to visitors/visits info
//
// Slight Revision by WhiteStar Magic (July 23 2009)
//  - Modified to prevent from capturing Owner Visits as this skews total visit counts
//  - Added a Menu for the Owner to offer choice of Reset System, Display list data, Give list data
//  - Added that on Owner Change the system Resets the data notecard & description as well
//  - Added a Script reset for when a Region is restarted, it hung up a couple of times on restart.
//
// Bug fixes and improvements by Carn Iggle (30 May 2010)
//  - Fixed handling of menu dialog timeouts.
//  - Fixed removal of listen handler for menu dialog.
//  - Fixed display of visitor list such that all lines are dumped to the chat.
//  - Added start date of logging to visitors/visits info.
//  - Added tracing of logging start date and most recent visit date. (15 July 2010)
//  - Added option for logging the first and last visit dates for each visitor. (15 July 2010)
//
// Visual Board version with Sim Stats by Kahn Khalim (17 July 2010)
//  - Added use of the OSSL function "osSetDynamicTextureData".
//  - Added Visual display board utilizing prim texture drawing.
//  - Prim Texture Drawing is done with "OS Dynamic Texture Language", not "LSL Helper Functions".
//  - Added visual stats for Region FPS and Time Dilation.
//  - Added Visual list of Avatars(agents) currently on region.
//  - Added Health algorithm for region and condition indicator
//
//======================================== Variables ========================================

// ------------------------------ Configuration variables -----------------------------------

integer FLOATING_TEXT = FALSE;              // Show number of visitors also as floating text
integer LOG_VISITOR_DATES = TRUE;        // Log dates of first and last visit for each visitor
float TimerInterval = 60.0;                // Number of seconds between polling the region for new avatars
string DataNotecard = "Visitor List Data"; // Name of the notecard in which data is stored

// ------------------------- Internal variables - do not change -----------------------------

list     Visitors;      // Data stored in the format [ UUID, Name, Visits [...]]
integer  Stride;        // Length of an entry in Visitors list
list     CurrentlyHere; // UUIDs of avatars detected during the previous scan
key      DataRequest;   // Used for reading in the settings notecard
integer  NotecardLine;  // Likewise
string   LastVisit;     // Date of last visit
string   drawdata;      // String to hold the Texture Draw commands.

//======================================== Functions ========================================

string c(integer x, integer y, integer width,integer fontsize,string text,string commands){
    integer h = (width/2)  - (llStringLength( text ) * ((fontsize*2/3))/2);
    string dd = "MoveTo "+((string)(x+h))+","+(string)y+";FontSize ";
    dd += (string)fontsize+"; "+commands+"Text "+text+";";
    return dd;
}
initboard(){    //Draw board defaults
    drawdata += "PenSize 2;PenColour black; FillRectangle 512, 256;";
    drawdata += "MoveTo 10,10; PenColour red; Rectangle 492, 236;MoveTo 10,35;";
    drawdata += "LineTo 502,35;MoveTo 256,35;LineTo 256,251;PenColour blue;";
    drawdata += c(10,10,492,14,"Sim Stats on '"+llGetRegionName()+"' Region","FontProp B;");
    drawdata += "PenColour red;MoveTo 10,40;FontSize 12;FontProp B;Text Unique Visiters:;";
    drawdata += "PenColour red;MoveTo 10,65;FontSize 12;FontProp B;Text Total Visits:;";
    drawdata += "PenColour red;"+c(266,40,236,12,"Avatars Currently in Region","FontProp B;");
    drawdata += "PenColour red;"+c(10,110,236,12,"Sim F.P.S.","FontProp B;");
    drawdata += "MoveTo 23,130; PenColour yellow; Rectangle 220, 20;";
    drawdata += "PenColour red;"+c(10,155,236,12,"Sim Time Dilation","FontProp B;");
    drawdata += "MoveTo 23,175; PenColour yellow; Rectangle 220, 20;";
    drawdata += "PenColour red;"+c(10,200,236,12,"Sim Overall State","FontProp B;");
}
fillboard(){     // Write data to board
        drawdata="";
        initboard();
        integer VisitorsLogged = llGetListLength( Visitors ) / Stride;
        integer visits = CountTotalVisits();
        string ss1 = "s";
        string ss2 = "s";
        if (VisitorsLogged == 1) ss1 = "";
        if (visits == 1) ss2 = "";
        drawdata += "PenColour yellow;MoveTo 180,40;FontSize 12;FontProp B;Text ";
        drawdata += (string)VisitorsLogged+";";
        drawdata += "PenColour yellow;MoveTo 180,65;FontSize 12;FontProp B;Text ";
        drawdata += (string)visits+";";
        integer i;
        integer p;
        for(i=0;i<=llGetListLength( CurrentlyHere );i++){
            p = (i*25)+65;
            drawdata += "PenColour yellow;MoveTo 296,"+(string)p+";FontSize 12;FontProp B;Text ";
            drawdata += llKey2Name(llList2String( CurrentlyHere, i ))+";";
        }
        integer fps = (llRound((llGetRegionFPS() * 2.0) + 10.0) / 10);
        integer dilation = llRound((llGetRegionTimeDilation() * 10.0));
        integer combo = (dilation + fps);
        drawdata += "MoveTo 23,130; PenColour yellow; FillRectangle "+(string)(fps*15)+", 20;";
        drawdata += "PenColour gray;MoveTo 25,130;FontSize 12;FontProp B;Text ("+(string)llRound(llGetRegionFPS())+" FPS);";
        drawdata += "MoveTo 23,175; PenColour yellow; FillRectangle "+(string)(dilation*22)+", 20;";
        drawdata += "PenColour gray;MoveTo 25,175;FontSize 12;FontProp B;Text ("+(string)llGetRegionTimeDilation()+" DILATION);";
    drawdata += "PenColour yellow;"+c(10,225,206,12,GetCondition(combo*2),"FontProp B;");
        osSetDynamicTextureData("","vector", drawdata,"width:512,height:256", 0);
} 
string GetCondition(integer c)  //Sim Health meter
{                             
    if(c >= 40)
    return "AWESOME";
    else if(c > 35)
    return "ATHLETIC";
    else if(c > 30)
    return "HEALTHY";
    else if(c > 25)
    return "AVERAGE";
    else if(c > 20)
    return "ILL";
    else if(c > 15)
    return "DYING";
    else if(c > 10)
    return "CRITICAL";
    else
    return "DEAD?";
}
// AddName() adds the provided details to the master list, or updates the visit count if the person is
//           already present in the list.
// 
integer AddName( string UUID, string Name )
{
    integer TotalVisits = 1;
    string timestamp = GetTimestamp();
    integer IndexPos = llListFindList( Visitors, [ UUID ] ); // Search for the UUID in our master list

    if (IndexPos == -1) // If not found in the list, add them with 1 visit
    {
        if (LOG_VISITOR_DATES)
            Visitors += [ UUID, Name, (string) 1, timestamp, timestamp ];
        else         
            Visitors += [ UUID, Name, (string) 1 ];
    }
    else // If they were found in the list, increment their visit count, and update last visit date
    {
        integer CountPos = IndexPos + 2;
        TotalVisits = llList2Integer( Visitors, CountPos ) + 1;
        if (LOG_VISITOR_DATES)
        {
            string FirstVisit = llList2String( Visitors, CountPos+1 );
            Visitors = llListReplaceList( Visitors, [ (string) TotalVisits, FirstVisit, timestamp ], CountPos, CountPos + 2);
        }
        else
            Visitors = llListReplaceList( Visitors, [ (string) TotalVisits ], CountPos, CountPos );
    }
    return 1;
}
// CheckNames() gets the list of agents present in the region, ignores any who were present during the last scan,
//              and adds the remainder to the master list (or increments their visit count if they were already in it)
//
CheckNames()
{
    integer NamesAdded = 0;
    list l = osGetAgents();  // Get the list of all the agents present in the region (better than sensors!)
    list RecentScan;         // To hold the name and UUID of everyone found, but who was not present during last scan
    list RecentUUIDs;        // To hold just the UUIDs of everyone found, but who was not present during last scan
    list AllUUIDsFound;      // To hold the UUIDs of everyone found, regardless of whether or not they were present last time
    integer i = llGetListLength( l ); // general iterator variable
    //
    // First, build a list containing the data of everyone who wasn't found during the previous scan
    //
    while (i--)
    {
        string AvatarName = llList2String( l, i );                     // Get the full name
        list NameParts = llParseString2List( AvatarName, [ " "], [] ); // break it into first and last names
        key UUID = osAvatarName2Key( llList2String( NameParts, 0 ), llList2String( NameParts, 1 )); // Get the UUID
        AllUUIDsFound += [ UUID ];                                     // Add their UUID to the list of all UUIDs found
        if(UUID != llGetOwner()) {                                     // Do not count visits of the owner
            if (llListFindList( CurrentlyHere, [ UUID ] ) == -1)       // if we didn't find the person during the previous scan:
            {
                RecentUUIDs += [ UUID ];  // Add their UUID to the list of new UUIDs found during this scan
                RecentScan += [ UUID, AvatarName ];// Add all their information to the recent scan list
            }
        }
    }
    CurrentlyHere = AllUUIDsFound; // Replace CurrentlyHere with the list of everyone who's present in the region now.
    //
    // Add anyone new from the most recent scan to the master list, or increase their visit counts if they've already been here before
    //
    for (i = llGetListLength( RecentScan ); i > 0; i -= 2)
    {
        string thisUUID = llList2String( RecentScan, i-2 );
        NamesAdded += AddName( thisUUID, llList2String( RecentScan, i-1 ));
    }
    //
    // If we actually did anything, write out a new notecard to save the data.
    //
    if (NamesAdded)
    {
        LastVisit = GetTimestamp();
        if (llGetInventoryKey( DataNotecard ) != NULL_KEY) llRemoveInventory( DataNotecard );
        osMakeNotecard( DataNotecard, Visitors );
        llSetObjectDesc( GetInfoText() );
    }
    if (FLOATING_TEXT)
        llSetText( GetInfoText(), <1.0,1.0,1.0>, 1.0 );
}
// CountTotalVisits() returns the total number of visits recorded by all avatars
//
integer CountTotalVisits()
{
    integer RetVal = 0;
    integer i = llGetListLength( Visitors ) - 1 - (Stride - 3);
    for (; i > 0; i -= Stride)
    {
        RetVal += llList2Integer( Visitors, i );
    }
    return RetVal;
}
// Returns a formatted version of llGetTimestamp()
//
string GetTimestamp()
{
    list tokens = llParseString2List(llGetTimestamp(), ["T",".","Z"], []);
    return llList2String(tokens, 0);
}   

// Returns text with info about the scanning period
string GetDateInfo()
{
    string timeinfo = "";
    list tokens = llParseString2List(llGetObjectDesc(), [""], [" on ", " since ", " between ", " and "]);
    integer numparts = llGetListLength(tokens);
    string start = llList2String(tokens, -1);
    if (numparts >= 5)
    {
        start = llList2String(tokens, -3);
        if (LastVisit == "")
            LastVisit = llList2String(tokens, -1);
    }
    if (numparts >= 3)
    {
        if (LastVisit == "")
            timeinfo = "since " + start;
        else
            timeinfo = "between " + start + " and " + LastVisit;
    }
    return timeinfo;
}
// GetInfoText() returns an info text about the recorded number of visitors, total number of visits,
// and the date of the first and last visit.
//
string GetInfoText()
{
    integer VisitorsLogged = llGetListLength( Visitors ) / Stride;
    integer visits = CountTotalVisits();
    string ss1 = "s";
    string ss2 = "s";
    if (VisitorsLogged == 1) ss1 = "";
    if (visits == 1) ss2 = "";
    return (string) VisitorsLogged + " Visitor" + ss1 + " / " + (string) visits + " Visit" + ss2
        + " in " + llGetRegionName() + " " + GetDateInfo();
}

// DeliverReportNotecard() builds a notecard containing the visitor list sorted three ways: by frequency of visit (descending),
//                         alphabetically by visitor name, and by date of first visit. It then delivers the notecard to the
//                         owner, and deletes it from inventory.
//
// mod by WS string action for either display or give
//
DeliverReportNotecard(string action)
{
    list OrderedList;                         // to store the list ordered by date of visit
    list NameList;                            // to store the list ordered by avatar name
    list VisitList;                           // to store the list ordered by visit count
    list Notecard;                            // to store the report notecard
    integer j = llGetListLength( Visitors );
    integer i;
    integer VisitorsLogged = j / Stride;
    integer VisitsLogged = CountTotalVisits();
    llOwnerSay( (string) (j / Stride) + " visitors/" + (string) VisitsLogged + " visits logged. Please wait while the full report is generated." );
    for (i = 0; i < j; i += Stride)
    {
        string name    = llList2String( Visitors, i+1 );
        integer visits = llList2Integer( Visitors, i+2 );
        string dates   = "";
        string sss     = "s";
        if (visits == 1) sss = "";
        if (LOG_VISITOR_DATES)
            dates = " between " + llList2String( Visitors, i+3 ) + " and " + llList2String( Visitors, i+4 );
        OrderedList += [ llToLower( name ), name + " (" + (string) visits + " visit" + sss + dates + ")" ];
        VisitList += [ visits, (string) visits + dates + ": " + name ];
    }
    if (VisitorsLogged == 1) // OpenSim screws up llList2ListStrided if there's only one stride in the list.
    {
        OrderedList = llDeleteSubList( OrderedList, 0, 0 );
        NameList = OrderedList;
        VisitList = llDeleteSubList( VisitList, 0, 0 );
    }
    else
    {
        NameList = llList2ListStrided( llDeleteSubList( llListSort( OrderedList, 2, TRUE), 0, 0 ), 0, -1, 2 ); // sort alphabetically
        OrderedList = llList2ListStrided( llDeleteSubList( OrderedList, 0, 0 ), 0, -1, 2 );
        VisitList = llListSort( VisitList, 2, FALSE );                                   // sort by descending visit count
        VisitList = llList2ListStrided( llDeleteSubList( VisitList, 0, 0 ), 0, -1, 2 );  // Get only the text to display
    }
    Notecard =  [ "Total visitors: " + (string) VisitorsLogged ];
    Notecard += [ "Total visits: " + (string) VisitsLogged ];
    Notecard += [ "", "Sorted by number of visits: ", "" ] + VisitList;
    Notecard += [ "", "Sorted by name:", "" ] + NameList;
    Notecard += [ "", "Sorted by date of first visit:", "" ] + OrderedList;
    Notecard += [ "", "Total visitors: " + (string) VisitorsLogged ];
    Notecard += [ "Total visits: " + (string) VisitsLogged ];
    string NotecardName = llGetRegionName() + " visitor list " + llGetTimestamp() + ", " + GetDateInfo();
    if(action == "Display")
    {
        string name = llGetObjectName();
        llSetObjectName("");
        llOwnerSay("/me " + NotecardName);
        j = llGetListLength(Notecard);
        for (i = 0; i < j; ++i)
            llOwnerSay("/me " + llList2String(Notecard, i));
        llSetObjectName(name);
    }
    else
    {
        osMakeNotecard( NotecardName, Notecard );
        llGiveInventory( llGetOwner(), NotecardName );
        llRemoveInventory( NotecardName );
    }

}
// Modifications by WhiteStar follow
integer CHANNEL;
integer HANDLE;
MENU_Owner(key id)  // Admin Menu
{
    llListenRemove(HANDLE);                         // Safety kill listener incase left over
    CHANNEL = (integer)(llFrand(-1000.0) - 1000.0); // Random negative channel
    HANDLE = llListen(CHANNEL, "", NULL_KEY, "");   // listen for dialog answers
    //
    llDialog(id,
        "\nDisplay: List visitor data in chat" +
        "\nNotecard: Give a notecard with visitor data" +
        "\nReset: Deletes notecard and resets description",
        ["Notecard","Reset","CANCEL","Display"], CHANNEL);
    llSetTimerEvent(45.0); // to AutoKill Listener
}   

default
{
    state_entry()
    {
        llSetText( "", ZERO_VECTOR, 0.0 );
        Stride = 3;
        if (LOG_VISITOR_DATES) Stride = 5;
        Visitors = [];                                                    // clear out the visitors list
        if (llGetInventoryKey( DataNotecard ) == NULL_KEY) state running; // switch to running state if the notecard doesn't exist
        llOwnerSay( "Reading saved visitor list data" );
        NotecardLine = 0;
        DataRequest = llGetNotecardLine( DataNotecard, 0 );

    }
    dataserver( key id, string data )
    {
        if (id != DataRequest) return;
        if (data == EOF)
            state running;
        else
        {
            data = llStringTrim( data, STRING_TRIM );
            if (data != "") Visitors += [ data ];
            DataRequest = llGetNotecardLine( DataNotecard, ++NotecardLine );
        }
    }
}

// ================================ running state handler ==================================

state running
{
    // state_entry(): load in notecard, if present, set a timer, and check region immediately.
    //
    state_entry()
    {
        llOwnerSay( "Visitor scanning active" );
        llSetTimerEvent( TimerInterval );  // Set the timer for periodic checks
        CheckNames();                      // Check the region immediately for avatars
        fillboard();                       // Write Data to the display board
    }

    // touch_start() will cause the report card to be generated, if the touch was from our
    //               owner; otherwise, it will tell the toucher how many visitors the region
    //               has logged.
    //
    touch_start( integer foo )
    {
        if (llDetectedKey( 0 ) == llGetOwner()) // being lazy, and assuming there's only one touch.
        {
            MENU_Owner(llGetOwner());
        }
        else
        {
            llInstantMessage( llDetectedKey( 0 ), GetInfoText() );
        }
    }

    // listen event to capture owners menu response and act upon it.
    //
    listen( integer channel, string name, key id, string msg )
    {
        llListenRemove(HANDLE); HANDLE = 0; // Kill listen reduce Lag
        if(msg == "Reset")
        {
            llRemoveInventory(DataNotecard);
            llSetObjectDesc("Reset on " + GetTimestamp());
            llResetScript();
        }
        else if((msg == "Display")||(msg == "Notecard"))
            DeliverReportNotecard(msg);
    }

    // Poll the region every x seconds (as configured by the TimerInterval variable). Rather than using
    // sensors, which are limited both in range and in the number of avatars they can detect, we'll use
    // the OpenSim-only function osGetAgents().
    //
    timer()
    {
        if (HANDLE) {
            llListenRemove(HANDLE); HANDLE = 0;
            llSetTimerEvent( TimerInterval );
        }
        CheckNames();
        fillboard();
    }

    // Reset the script if the region has restarted. This mainly restarts the timer and clears out the list
    // of people who were detected the last time the region was scanned, but it's good to avoid any other
    // unforeseen consequences.
    //
    changed( integer change )
    {
        if (change & CHANGED_REGION_RESTART)
            llResetScript();
        if (change & CHANGED_OWNER) {
            llRemoveInventory(DataNotecard);
            llSetObjectDesc("Initialized on " + GetTimestamp());
            llResetScript();
        }
    }
}