/* ******************************************************************************************************************************
  File: Game GreedyZilch V2
  Author: Seth Nygard @ osgrid
  Date:   2014-06-25

  A complete Greedy/Zilch game in a single script.  Hopefully not too hard on resources.

  Note:  I have observed times were it appears that a die is clicked and the scores do not appear to update.  All testing so far
         has shown that the die was scored, the score panels updated, but the viewer failed to show the change in texture.
****************************************************************************************************************************** */
integer   DEBUG_MODE           = FALSE;
string    GAME                 = "Greedy Zilch";
integer   gGreedyGameChannel   = -443276;
integer   gFinalScore          = 10000;               // Final score to match or beat to end the game
key       gCurrentPlayerID     = NULL_KEY;            // UUID for curently active player avatar
integer   gCurrentPlayerIdx    = -1;                  // Index for curently active player avatar
string    gCurrentPlayerName   = "";                  // Name for curently active player avatar
integer   gCurrentPlayerScore  = 0;                   // Accumulated score for curently active player avatar
integer   gNumberOfPlayers     = 0;                   // Total number of players
integer   gFinalRound          = FALSE;               // True if final round (someone was >=10000)
integer   gLastPlayer          = -1;                  // Set to player index if player has reached end of game score (10000)
integer   gPassFlag            = FALSE;               // True if play is being passed to another player
list      gPlayerIDList        = [];                  // Key ID for the player
list      gPlayerNameList      = [];                  // Names for players
list      gPlayerScoreList     = [];                  // Total score for players
list      gPlayerOnBoard       = [];                  // TRUE if player has scored 500 points in a single round, is on the baord
list      gPlayerPosition      = [];                  // Number of position that player is seated at: 0-7
integer   gWinner              = FALSE;               // True if we have a winner
integer   gPlaying             = FALSE;               // True if game play is active
integer   gTurnScore           = 0;                   // Accumulated score for the current turn
integer   gRollScore           = 0;                   // Score for the current die roll
integer   gOldRollScore        = 0;                   // Previous roll score, used to control when we update the score display
integer   gHighScoreIdx        = -1;                  // Index of player with highest scrore
integer   gHighScore           = 0;                   // Value of highest score
integer   gExpectedScore       = 0;                   // Calculated value of dice in current roll
integer   gTableMode           = 0;                   // Current mode for the table
integer   gTimerWait           = FALSE;               // Used for delay timer, TRUE if waiting
integer   gDie                 = 0;                   // Integer containing die values mapped to each nibble, high bit is 1 if locked, bits 24-30 are 1 if die is picked
integer   gBadDieCount         = 0;                   // Count of selected but unscoring dice
integer   gDieScoredCount      = 0;                   // Count of die with values
integer   gDieLockedCount      = 0;                   // Count of scored/locked die
integer   gRollCount           = 0;                   // Number of rolls in a player's round, used to allow 500 points on Zilch for first roll
list      gDieTextureList      = [ "die_face_1", "die_face_2", "die_face_3", "die_face_4", "die_face_5", "die_face_6" ];
list      gDieSideList         = [ 0x142530, 0x502431, 0x051342, 0x054213, 0x052134, 0x402135 ];
list      gDiceLinkID          = [ 12, 13, 14, 15, 16, 17 ];
list      gStoolLinkID         = [ 25, 24, 23, 22, 21, 20, 19, 18 ];
list      gPlayerScoreLinkID   = [ 9, 8, 7, 6, 5, 4, 3, 2 ];
list      gPlayButtonsLinkID   = [ 27, 28 ];
integer   gRoundScoreLinkID    = 10;
integer   gTotalScoreLinkID    = 11;
integer   gMainButtonsLinkID   = 28;
vector    RED                  = <1.0, 0.0, 0.0>;
vector    GREEN                = <0.2, 1.0, 0.2>;
vector    WHITE                = <1.0, 1.0, 1.0>;
vector    YELLOW               = <1.000, 0.863, 0.000>;
vector    BLUE                 = <0.2, 0.2, 1.0>;
vector    AQUA                 = <0.498, 0.859, 1.000>;
vector    LIME                 = <0.004, 1.000, 0.439>;
vector    SILVER               = <0.867, 0.867, 0.867>;
string    SIT_JOIN             = "SIT_JOIN";
string    SIT_LEAVE            = "SIT_LEAVE";
string    CMD_EJECT            = "CMD_EJECT";
string    CMD_RESET            = "CMD_RESET";
string    hexc                 = "0123456789ABCDEF";

/* **********************************************************************************
  MISC HELPER FUNCTIONS
********************************************************************************** */
string int2hex(integer x)
{
    integer x0 = x & 0xF;
    string res = llGetSubString(hexc, x0, x0);
    x = (x >> 4) & 0x0FFFFFFF; //otherwise we get infinite loop on negatives.
    while( x != 0 )
    {
        x0 = x & 0xF;
        res = llGetSubString(hexc, x0, x0) + res;
        x = x >> 4;
    }
    return res;
}


Debug(string msg)
{
    llOwnerSay("["+llGetScriptName()+"] "+msg);
}


integer GetLinkNumberByName(string name)
{
   integer i;
   integer n = llGetNumberOfPrims();
   string t;
   for (i=0;i<=n;i++) {
     t = llGetLinkName(i);
     if (DEBUG_MODE)  Debug( "Linkset ID = "+(string)i+" - "+t );
     if (name == t)
       return i;
   }
   return -1;
}

UnsitAllAvatars()
{
    integer c = llGetObjectPrimCount(llGetKey());
    integer n = llGetNumberOfPrims();
    for (; c < n; --n) {
        llUnSit(llGetLinkKey(n));
    }
}


/* **********************************************************************************
  DIE HANLDER FUNCTIONS
********************************************************************************** */
SetDieValue(integer NumDie, integer ValueDie)
{
    // We are using Integer weighted values to track each die as the list has proven unreliable in some cases and we have no arrays
    if(DEBUG_MODE) Debug( "SetDieValue: Die = "+int2hex(gDie) );
    integer posMask = ~(0x0F <<(NumDie*4));
    gDie = gDie & posMask;
    gDie += ValueDie << (NumDie*4);
    if(DEBUG_MODE) Debug( "SetDieValue - Mask="+int2hex(posMask)+" Die #"+(string)NumDie+"="+(string)ValueDie+" -> "+int2hex(gDie));
}


integer GetDieValue(integer dieNumber)
{
    return (gDie >> (dieNumber*4)) & 0x0F;                                         
}


SelectDie(integer dieNumber)
{
    gDie = gDie ^ (1 << (24+dieNumber));
    integer picked =  (((gDie>>24)&0x3F) & (1<<dieNumber)) > 0;

    if (picked) {
        SetDieColor( GREEN, TRUE, dieNumber );
    } else {
        SetDieColor( WHITE, TRUE, dieNumber );
    }
    if(DEBUG_MODE) Debug( "SelectDie: dieNumber = "+(string)dieNumber+"   picked = "+(string)picked+"    Die -> "+int2hex(gDie) );
}


// Set each face to show current die value, top side (face #1) is current roll value
SetDieTextures(integer dieNumber, integer val)
{
    integer ix;
    integer linkID = llList2Integer(gDiceLinkID, dieNumber);
    integer faces = llList2Integer(gDieSideList, val-1);
//    if(DEBUG_MODE) Debug( "SetDieTexture:  Die # = "+(string)dieNumber+"   Value = "+(string)val+"    Faces = "+int2hex(faces) );
    for(ix = 1; ix < 7; ix++) {
        integer face = (faces & 0x07);   //The mesh die has 7 faces, 1-6 are die faces, 0 is die body
        faces = (faces >> 4) & 0x00FFFFFF;
        string textureID = llList2String(gDieTextureList, face);
        llSetLinkPrimitiveParamsFast( linkID, [PRIM_TEXTURE, ix, (string)textureID, <1.0,1.0,0.0>, ZERO_VECTOR, 0.0 ] );
    }
}


SetDieColor(vector color,integer visible, integer dieNumber)
{
    integer linkID = llList2Integer(gDiceLinkID, dieNumber);
    llSetLinkPrimitiveParamsFast( linkID, [PRIM_COLOR, ALL_SIDES, color, (float)visible] );
}


ColorizeTheDice(vector color, integer visible)
{
    integer idx;
    for(idx=0;idx<6;idx++) {
      SetDieColor( color, visible, idx );
    }
}


ClearTheDice()
{
    gDie = 0;
    gDieScoredCount = 0;
    gDieLockedCount = 0;
    ColorizeTheDice( WHITE, TRUE );  // Show all die in white
}


RollTheDice()
{
    integer dieNum;
    if (DEBUG_MODE) Debug( "RollTheDice:  Die -> "+int2hex(gDie) );
    if ( (gDieScoredCount+gDieLockedCount==6) && (gBadDieCount==0) )   ClearTheDice();
    for(dieNum=0;dieNum<6;dieNum++) {
      integer picked = (((gDie>>24)&0x3F) & (1<<dieNum)) > 0;
      integer locked = (gDie & (1<<(dieNum*4)+3)) > 0;
      if (DEBUG_MODE) Debug( "RollTheDice:  die #"+(string)dieNum+"   Locked = "+(string)locked+"   Picked = "+(string)picked );
      if ( picked ) {
          gDie += 1 << (3+(dieNum*4));
          SetDieColor( BLUE, TRUE, dieNum );
      } else if ( !locked ) {
        integer n = GetDieValue(dieNum) & 0x07;
        integer OldValue = n;
        while( n == OldValue ) {
            n = ((integer)(llFrand(6.0)+1));
        }
        if (DEBUG_MODE) Debug( "    die "+(string)dieNum+"  has value "+(string)n );
        SetDieTextures(dieNum,n);
        SetDieValue(dieNum,n);
      }
    }
    gDie = gDie & 0x00FFFFFF;
    gOldRollScore = 0;
    gRollCount++;
    if (DEBUG_MODE) Debug( "RollTheDice: -> "+int2hex(gDie) );
}


/* **********************************************************************************
  SCORING AND GAME PLAY FUNCTIONS
********************************************************************************** */
integer CheckPoints(integer allDie)
{
    integer valuedCount = 0;
    integer DieCounts= 0;
    gBadDieCount = 0;
    gDieScoredCount = 0;
    gDieLockedCount = 0;
    integer ix = 0;
    integer selDice = gDie;

    if(DEBUG_MODE) Debug( "CheckPoints called: Dice = "+int2hex(gDie)+"   AllDice="+(string)allDie );
    for(ix = 0; ix < 6; ix++) {
        integer picked = (((gDie>>24)&0x3F) & (1<<ix)) > 0;
        integer locked = (gDie & (1<<(ix*4)+3)) > 0;
        integer die = selDice & 0x0F;
        selDice = (selDice >> 4) & 0x00FFFFFF;
        if (DEBUG_MODE) Debug( "CheckPoints:  die #"+(string)ix+"   Locked = "+(string)locked+"   Picked = "+(string)picked );
        if ( !locked ) {
          if ( allDie || picked ) {
            DieCounts += 1 << ((die-1)*4);
            valuedCount++;
            gDieScoredCount += picked;
          }
        } else {
            gDieLockedCount++;
        }
    }

    if(DEBUG_MODE) Debug( "CheckPoints:-> Counts = "+int2hex(DieCounts) );
    integer points = 0;
    ix = 0;
    integer pairs = 0;
    integer n = DieCounts;
    for(ix = 0; ix < 6; ix++) {
        pairs += (n & 0x0f) == 2;
        n = (n & 0x00FFFFFF) >> 4;
    }

    if(pairs >= 3) {
        //We found 3 pairs so 1500 points
        points = 1500;
    } else if( DieCounts == 0x111111 ) {
        //We found a straight so 3000 points
        points = 3000;
    } else {
        integer dieCnt = 0;
        integer dieFace = 0;
        ix = 0;
        n = DieCounts;
        for( ix = 0; ix < 6; ix++) {
            dieFace = ix + 1;

            dieCnt = n & 0x0F;
            n = (n & 0x00FFFFFF) >> 4;

            if(dieCnt == 1) {
                // Single die found with value, check if it is a 1 or a 5
                if(dieFace == 1) {   // each 1 adds 100 points
                    points += 100;
                } else if(dieFace == 5) {     // each 5 adds 50 points
                    points += 50;
                } else {      // singles of any other die adds nothing and is an incorrectly picked die
                    gBadDieCount += 1;
                }
            } else if(dieCnt == 2) {
                // We found a pair, check if it is a 1 or a 5
                if(dieFace == 1) {
                    points += 200;
                } else if(dieFace == 5) {
                    points += 100;
                } else {
                    gBadDieCount += 2;
                }
            } else if(dieCnt == 3) {
                // We found 3 of a kind
                if(dieFace == 1) {
                    points += 1000;
                } else {
                    points += (dieFace * 100);
                }
            } else if(dieCnt == 4) {
                // We found 4 of a kind!
                if(dieFace == 1) {
                    points += 2000;
                } else {
                    points += (dieFace*2)*100;
                }
            } else if(dieCnt == 5) {
                // We found 5 of a kind!!
                if(dieFace == 1) {
                    points += 4000;
                } else {
                    points += (dieFace*4) * 100;
                }
            } else if(dieCnt == 6) {
                // We found 6 of a kind!!!!
                if(dieFace == 1) {
                    points = 8000;
                } else {
                    points = (dieFace*8) * 100;
                }
                ix = 7;
            }
        }

        // If we roll a Zilch on the first roll then we can collect 500 points and continue the round
        if( (valuedCount==6) && (points==0) && (gRollCount < 2) ) {
            gBadDieCount = 0;
            gDieScoredCount = 6;
            points=500;
            if(DEBUG_MODE) Debug( "CheckPoints: Zilch on first roll = 500 points" );
        }
    }
    if(DEBUG_MODE) Debug( "[CheckPoints] Accumulated points = "+(string)points+"    Scored Dice = "+(string)gDieScoredCount+"   # with value="+(string)valuedCount+"  Non-scoring="+(string)gBadDieCount+"   Locked="+(string)gDieLockedCount+"   RollCount="+(string)gRollCount );
    return points;
}


GamePlay(integer mode)
{
  integer points = CheckPoints(FALSE);
  if ( mode == -1 ) {  // Pass turn
      if (DEBUG_MODE)  Debug( "PASS clicked, first roll" );
      if( points > 0 ) {
          if( gBadDieCount > 0 ) {
              llWhisper(0,"You seem to have selected "+(string)gBadDieCount+" die that have no value.  You need to unselect them.");
          } else {
              gTurnScore += points;
              if( points < gExpectedScore )
                  llWhisper(0,"Oops!  You missed "+(string)(gExpectedScore-points)+" points on that round.");
          }
          gCurrentPlayerScore += gTurnScore;
      }
      ColorizeTheDice( WHITE, FALSE );  // Hide all the dice
      PassTurn();
  } else {
    if ( gRollCount < 1 ) {    // First roll
        if (DEBUG_MODE)  Debug( "ROLL clicked, first roll" );
        ClearTheDice();
        llPlaySound("dice roll", 1);
        RollTheDice();
    } else {  // Subsequent rolls
        if (DEBUG_MODE)  Debug( "ROLL clicked, next roll." );
        if( points > 0 ) {
            if( gBadDieCount > 0 ) {
                llWhisper(0,"You seem to have selected "+(string)gBadDieCount+" die that have no value.  You need to unselect them.");
            } else {
                gTurnScore += points;
                if( points < gExpectedScore )
                    llWhisper(0,"Oops!  You missed "+(string)(gExpectedScore-points)+" points on that round.");

                if( gDieScoredCount >= 6 ) {
                    gDieScoredCount = 0;
                    if(DEBUG_MODE) Debug("We have 6 scored dice so we reset them all and roll again");
                    ClearTheDice();
                }
                llPlaySound("dice roll", 1);
                RollTheDice();
            }
        }
    }
    gExpectedScore = CheckPoints(TRUE);
    if( gExpectedScore == 0 ){
        ColorizeTheDice( YELLOW, TRUE );  // Show all die in yellow
        llPlaySound("Buzzer", 1);
        llWhisper(0,"Sorry, there are no points in this roll and you lose all points from this round.");
        gTurnScore = 0;
        llSleep(2.0);
        PassTurn();
    }
  }
}


StartGame()
{
    integer i;
    gFinalRound = FALSE;
    gWinner = FALSE;
    gHighScore = 0;
    gHighScoreIdx = -1;
    gCurrentPlayerScore = 0;
    gPlayerScoreList = [];
    gPlayerOnBoard = [];
    if ( gNumberOfPlayers > 0 ) {
      for(i=0;i<gNumberOfPlayers;i++) {
          gPlayerScoreList += [0];
          gPlayerOnBoard += [0];
      }
      gPlaying = TRUE;
      SetTableMode(2);
      UpdateScores();
    }
}


EndGame()
{
    integer i;
    gWinner = TRUE;
    UpdateScores();
    SetTableMode(3);
    llPlaySound("Ta Da", 1);
    for( i=0; i< gNumberOfPlayers; i++ ) {
        integer s = llList2Integer(gPlayerScoreList,i);
        if( s > gHighScore ) {
            gHighScore = s;
            gHighScoreIdx = i;
        }
    }
    gCurrentPlayerID = llList2Key(gPlayerIDList, gHighScoreIdx);
    gCurrentPlayerName = llKey2Name(gCurrentPlayerID);
    UpdateScorePanel(0,gRoundScoreLinkID);
    UpdateScorePanel(gHighScore,gTotalScoreLinkID);
    llWhisper(0,gCurrentPlayerName+" has won the game with "+(string)gHighScore+" points!");
    llSay(gGreedyGameChannel,"GreedyZilch|"+(string)gHighScore+"|"+gCurrentPlayerName );
    llSetTimerEvent(10);
}



PassTurn()
{
    PickNextPlayer();
    ColorizeTheDice( WHITE, FALSE );  // Hide all the dice
    gTurnScore = 0;
    gPassFlag = FALSE;
    gRollCount = 0;
    gDie = 0;
    UpdateScorePanel(gTurnScore,gRoundScoreLinkID);
    UpdateScorePanel(gCurrentPlayerScore,gTotalScoreLinkID);
}


UpdateLinkIDLists()
{
    integer i = llGetNumberOfPrims();

    for (; i > 0; i--) {
        string  ObjectName  = llGetLinkName(i);
        list    object  = llParseString2List(ObjectName,[":"],[""]);
        string  objType = llList2String(object,0);
        integer objNum  = llList2Integer(object,1)-1;

        if (DEBUG_MODE) Debug( "UpdateLinkIDLists: #"+(string)i+" ObjectName="+ObjectName+" ObjectType="+objType+" ObjectNumber="+(string)objNum );

        if ( objType == "Die" && objNum >= 0 && objNum < 6 ) {
            gDiceLinkID = llListReplaceList(gDiceLinkID,[i],objNum,objNum);
        } else if ( objType == "TScore" && objNum >=0 && objNum < 8 ) {
            gPlayerScoreLinkID = llListReplaceList(gPlayerScoreLinkID,[i],objNum,objNum);
        } else if ( objType == "Stool" && objNum >=0 && objNum < 8 ) {
            gStoolLinkID = llListReplaceList(gStoolLinkID,[i],objNum,objNum);
        } else if ( objType == "Buttons" && objNum >=0 && objNum < 2 ) {
            gPlayButtonsLinkID = llListReplaceList(gPlayButtonsLinkID,[i],objNum,objNum);
        } else if ( objType == "PScore" ) {
            gTotalScoreLinkID    = i;
        } else if ( objType == "RScore" ) {
            gRoundScoreLinkID    = i;
        } else if ( objType == "MainButtons" ) {
            gMainButtonsLinkID = i;
        }
    }

    if (DEBUG_MODE) {
      llOwnerSay( "Dice: "+llDumpList2String(gDiceLinkID,", ") );
      llOwnerSay( "Scores: "+llDumpList2String(gPlayerScoreLinkID,", ") );
      llOwnerSay( "Stools: "+llDumpList2String(gStoolLinkID,", ") );
      llOwnerSay( "Round score: "+(string)gRoundScoreLinkID );
      llOwnerSay( "Player score: "+(string)gTotalScoreLinkID );
      llOwnerSay( "Main Buttons: "+(string)gMainButtonsLinkID );
      llOwnerSay( "Play Buttons: "+llDumpList2String(gPlayButtonsLinkID,", ") );
    }
}


SetTableMode(integer mode)
{
  integer i;
  if ( mode == 0 ) { // Mode 0 = IDLE - NO AVATARS SITTING
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,0), 0.0, ALL_SIDES );
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,1), 0.0, ALL_SIDES );
      llSetLinkAlpha( gMainButtonsLinkID, 0.0, ALL_SIDES );
      llSetLinkAlpha( gRoundScoreLinkID, 0.0, ALL_SIDES );
      llSetLinkAlpha( gTotalScoreLinkID, 0.0, ALL_SIDES );
      ColorizeTheDice( WHITE, FALSE );  // Hide all the dice
      for(i=0;i<8;i++) {
        llSetLinkAlpha( llList2Integer(gPlayerScoreLinkID,i), 0.0, ALL_SIDES );
      }
  } else if ( mode == 1 ) { // Mode 1 = WAITING - At least 1 avatar is seated
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,0), 0.0, ALL_SIDES );
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,1), 0.0, ALL_SIDES );
      llSetLinkAlpha( gMainButtonsLinkID, 1.0, ALL_SIDES );
      llSetLinkAlpha( gRoundScoreLinkID, 0.0, ALL_SIDES );
      llSetLinkAlpha( gTotalScoreLinkID, 0.0, ALL_SIDES );
      ColorizeTheDice( WHITE, FALSE );  // Hide all the dice
  } else if ( mode == 2 ) { // Mode 2 = PLAY - Game play mode
      UpdateScorePanel(0,gRoundScoreLinkID);
      UpdateScorePanel(0,gTotalScoreLinkID);
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,0), 1.0, ALL_SIDES );
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,1), 1.0, ALL_SIDES );
      llSetLinkAlpha( gMainButtonsLinkID, 0.0, ALL_SIDES );
      llSetLinkAlpha( gRoundScoreLinkID, 1.0, ALL_SIDES );
      llSetLinkAlpha( gTotalScoreLinkID, 1.0, ALL_SIDES );
      ColorizeTheDice( WHITE, FALSE );  // Hide all the dice
  } else { // End game
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,0), 0.0, ALL_SIDES );
      llSetLinkAlpha( llList2Integer(gPlayButtonsLinkID,1), 0.0, ALL_SIDES );
      llSetLinkAlpha( gMainButtonsLinkID, 0.0, ALL_SIDES );
      llSetLinkAlpha( gRoundScoreLinkID, 0.0, ALL_SIDES );
      llSetLinkAlpha( gTotalScoreLinkID, 1.0, ALL_SIDES );
      ColorizeTheDice( WHITE, FALSE );  // Hide all the dice
  }
  gTableMode = mode;
}


/* **********************************************************************************
  PLAYER HANDLER FUNCTIONS
********************************************************************************** */
UpdateScorePanel(integer score,integer linknumber)
{
    integer digitNum = 1;
    integer pos;
    for(pos=1; pos<6; pos++) {
        integer digit = score - ( (integer)(score/(digitNum*10)) ) * (digitNum*10);
        digit = (integer)(digit / digitNum);
        digit = 360-(90*digit);
        float offset = ((float)digit) / 1000;
        llSetLinkPrimitiveParamsFast(linknumber,[PRIM_TEXTURE, pos, "numbers_t", <0.7,0.08,0.0>, <-0.05,offset,0.0>, 0.0]);
        digitNum *= 10;
    }
    if(DEBUG_MODE) Debug("Update score panel "+(string)linknumber+"  -> "+(string)score);
}


WelcomeNewPlayer(key id,integer pos)
{
  string playerName = llKey2Name(id);
  if( llListFindList(gPlayerIDList, [id]) == -1 ) {
    if( gNumberOfPlayers < 8 ) {
        ++gNumberOfPlayers;
        gPlayerIDList += [id];
        gPlayerScoreList += [0];
        gPlayerOnBoard += [FALSE];
        gPlayerNameList += [playerName];
        gPlayerPosition += [pos];
        UpdateScores();
//        integer link = llList2Integer(gPlayerScoreLinkID,pos);
//        llSetLinkAlpha( link, 1.0, ALL_SIDES );
//        llSetLinkAlpha( link, 0.0, 0 );
//        UpdateScorePanel( 0, link );

        llWhisper(0,  playerName + " has joined, welcome to "+GAME+".");
        llWhisper(0,  "There are now "+(string)gNumberOfPlayers+" joined in the game.");
    } else {
        llWhisper(0, "Sorry, there is only enough room for 8 players.");
    }
  } else {
    llWhisper(0, "Sorry "+playerName+", you can only play one position.  You have already joined the game.");
  }
  if ( gNumberOfPlayers > 0 && gTableMode == 0 ) {
    SetTableMode(1);
  }
}


RemovePlayer(key id)
{
  string playerName = llKey2Name(id);
  integer n = llListFindList(gPlayerIDList, [id]);
  if(DEBUG_MODE) Debug("Remove player "+(string)id+" requested.  List idx="+(string)n);
  if( n!= -1 ) {
      gPlayerIDList = llDeleteSubList(gPlayerIDList,n,n);
      gPlayerScoreList = llDeleteSubList(gPlayerScoreList,n,n);
      gPlayerOnBoard = llDeleteSubList(gPlayerOnBoard,n,n);
      gPlayerOnBoard = llDeleteSubList(gPlayerNameList,n,n);
      gPlayerPosition = llDeleteSubList(gPlayerPosition,n,n);
      --gNumberOfPlayers;
      llWhisper(0,  playerName + " has left the game.");
//      llSetLinkAlpha( llList2Integer(gPlayerScoreLinkID,llList2Integer(gPlayerPosition,n)), 0.0, ALL_SIDES );
      UpdateScores();
      llWhisper(0,  "There are now "+(string)gNumberOfPlayers+" joined in the game.");
      if(DEBUG_MODE){
        Debug("#Players = "+(string)gNumberOfPlayers+"   gPlayerIDList = "+llDumpList2String(gPlayerIDList," | "));
      }
  }
  if ( gNumberOfPlayers == 0 && gTableMode != 0 ) {
    llWhisper(0,  "Everyone has left the game.  Game Reset");
    SetTableMode(0);
    llResetScript();
  }
}


UpdateScores()
{
  integer i;
  if( (gCurrentPlayerIdx >= 0) ) {
      if( gTurnScore > 0 ) {
        if(!(llList2Integer(gPlayerOnBoard, gCurrentPlayerIdx))) {
            if(gTurnScore >= 500){
                gPlayerOnBoard = llListReplaceList(gPlayerOnBoard, [TRUE], gCurrentPlayerIdx, gCurrentPlayerIdx);
                gPlayerScoreList = llListReplaceList(gPlayerScoreList, [gCurrentPlayerScore], gCurrentPlayerIdx, gCurrentPlayerIdx);
                llWhisper(0,gCurrentPlayerName+" is now on the board.");
            }
        } else {
            gPlayerScoreList = llListReplaceList(gPlayerScoreList, [gCurrentPlayerScore], gCurrentPlayerIdx, gCurrentPlayerIdx);
        }
        if( gCurrentPlayerScore > gFinalScore  && !gFinalRound) {
            gFinalRound = TRUE;
            gLastPlayer = gCurrentPlayerIdx;
            llWhisper(0,gCurrentPlayerName+" has reached "+(string)gCurrentPlayerScore+" points.  This is the final round.  Each other player has one round, after which the highest score wins.");
        }
      }
  }

  for( i=0; i<gNumberOfPlayers; i++ ) {
        integer s = llList2Integer(gPlayerScoreList,i);
        if( s > gHighScore ) {
            gHighScore = s;
            gHighScoreIdx = i;
        }
  }

  for( i=0; i<8; i++ ) {
      vector c;
      float v=0;
      integer playerNum = llListFindList(gPlayerPosition,i);
      integer posLinkID = llList2Integer(gPlayerScoreLinkID,i);
      integer s = llList2Integer(gPlayerScoreList,playerNum);
      if ( gTableMode <= 1 ) {
        c = SILVER;
        if ( playerNum > -1 ) {
          v = 1;
        }
      } else {
        if ( playerNum > -1 ) {
          v = 1;
          if(!(llList2Integer(gPlayerOnBoard, playerNum))) {
            c = YELLOW;
          } else if( playerNum == gHighScoreIdx ) {
            c = GREEN;
          } else {
            c = AQUA;
          }
        }
      }
      llSetLinkPrimitiveParamsFast( posLinkID,
                                    [ PRIM_COLOR, 0, RED, 0,
                                      PRIM_COLOR, 1, c, v,
                                      PRIM_COLOR, 2, c, v,
                                      PRIM_COLOR, 3, c, v,
                                      PRIM_COLOR, 4, c, v,
                                      PRIM_COLOR, 5, c, v,
                                      PRIM_COLOR, 6, c, v ] );
      UpdateScorePanel( s, posLinkID );
      if (DEBUG_MODE) Debug( "UpdateScores: i="+(string)i+"  posLink="+(string)posLinkID+"  player="+(string)playerNum+"   c="+(string)c+"  v="+(string)v );
  }
}


PickNextPlayer()
{
    string txt;
    UpdateScores();
    if(gCurrentPlayerIdx == -1) {
        gCurrentPlayerIdx = (integer)llFrand((float)gNumberOfPlayers);
        llWhisper( 0, "Randomly picking player to start the game.");
    } else {
//        llSetLinkAlpha( llList2Integer(gPlayerScoreLinkID,llList2Integer(gPlayerPosition,gCurrentPlayerIdx)), 0.0, 0 );
        ++gCurrentPlayerIdx;
        if( gCurrentPlayerIdx >= gNumberOfPlayers )
            gCurrentPlayerIdx = 0;
    }

    gCurrentPlayerID = llList2Key(gPlayerIDList, gCurrentPlayerIdx);
    gCurrentPlayerScore = llList2Integer(gPlayerScoreList, gCurrentPlayerIdx);
    gCurrentPlayerName = llList2String(llParseString2List(llKey2Name(gCurrentPlayerID), [" "], []), 0);

    if( gFinalRound && gLastPlayer==gCurrentPlayerIdx) {
        EndGame();
    } else {
        llSetLinkAlpha( llList2Integer(gPlayerScoreLinkID,llList2Integer(gPlayerPosition,gCurrentPlayerIdx)), 1.0, 0 );
        txt = "It is " + gCurrentPlayerName + "'s turn.";
        integer onboard = llList2Integer(gPlayerOnBoard, gCurrentPlayerIdx);
        if( !onboard )
             txt +=  "You will need to accumulate 500 points this round to get on the board.";
        llPlaySound("Chime", 1);
        llWhisper( 0, txt);
    }
}


init()
{
    llWhisper(0,  GAME+" is initializing...");
    UpdateLinkIDLists();
    llMessageLinked( LINK_ALL_OTHERS, 0, CMD_EJECT, NULL_KEY );
    gCurrentPlayerIdx = -1;
    gNumberOfPlayers = 0;
    gPlayerIDList = [];
}


/* **********************************************************************************
  STATE MACHINES
********************************************************************************** */

default
{
    state_entry()
    {
        init();
        state IDLE;
    }

    link_message(integer sender_number, integer number, string message, key id)
    {
        if(DEBUG_MODE) Debug("LinkMessageRX Sender="+(string)sender_number+" |"+message+"| KeyID:"+(string)id);
        list msgParts = llParseString2List( message, [" "], [] );
        string cmd = llList2String( msgParts, 0 );
        string arg1 = llList2String( msgParts, 1 );
        integer num1 = (integer)arg1;

        if( cmd == CMD_RESET ) {
            UnsitAllAvatars();
            llMessageLinked( LINK_ALL_OTHERS, 0, CMD_RESET, NULL_KEY );
            llResetScript();
        } else if(cmd == SIT_LEAVE) {
            RemovePlayer(id);
        }
    }
    
    changed(integer change)
    {
      if ( change * CHANGED_LINK ) {
          UpdateLinkIDLists();
      }
    }
}


state IDLE
{
    state_entry()
    {
        integer i;
        gPlaying = FALSE;
        gCurrentPlayerScore = 0;
        gPlayerScoreList = [];
        gPlayerOnBoard = [];
        if ( gNumberOfPlayers > 0 ) {
          for(i=0;i<gNumberOfPlayers;i++) {
              gPlayerScoreList += [0];
              gPlayerOnBoard += [0];
          }
          SetTableMode(1);
        } else {
          SetTableMode(0);
        }
        UpdateScores();
        llWhisper(0,  GAME+" is ready.");
    }

    touch_start(integer num)
    {
        integer linkID      = llDetectedLinkNumber(0);
        integer faceID      = llDetectedTouchFace(0);
        key     avatarKey   = llDetectedKey(0);
        string  avatarName  = llDetectedName(0);
        string  ObjectName  = llGetLinkName(linkID);


        if (faceID == TOUCH_INVALID_FACE) {
            llOwnerSay("Sorry, your viewer doesn't support touched faces. You won't be able to play this game");
        } else {
            list    object  = llParseString2List(ObjectName,[":"],[""]);
            string  objType = llList2String(object,0);
            integer objNum  = llList2Integer(object,1);
            if ( objType == "MainButtons" && objNum == 0 ) {
               if ( faceID == 1 ) {       // RESET
                   llResetScript();
               } else if ( faceID == 3 ) {       // START
                  if(gNumberOfPlayers > 0) {
                      StartGame();
                      state PLAY;
                  }
               } else if ( faceID == 4 ) {       // HELP
                 llGiveInventory(llDetectedKey(0), "Greedy Zilch V2 Instructions");
               }
            } else if ( linkID == 1 ) {
                UpdateLinkIDLists();
            }
        }
    }


    link_message(integer sender_number, integer number, string message, key id)
    {
        if(DEBUG_MODE) Debug("LinkMessageRX Sender="+(string)sender_number+" |"+message+"| KeyID:"+(string)id);
        list msgParts = llParseString2List(message, [" "], []);
        string cmd = llList2String(msgParts, 0);
        string arg1 = llList2String(msgParts, 1);

        if(cmd == SIT_JOIN) {
            WelcomeNewPlayer(id,(integer)arg1-1);
        } else if(cmd == SIT_LEAVE) {
            RemovePlayer(id);
        } else if( cmd == CMD_RESET ) {
            UnsitAllAvatars();
            llMessageLinked( LINK_ALL_OTHERS, 0, CMD_RESET, NULL_KEY );
            llResetScript();
        }
    }

    changed(integer change)
    {
      if ( change * CHANGED_LINK ) {
          UpdateLinkIDLists();
      }
    }
}


state PLAY
{
    state_entry()
    {
        llWhisper(0, "Starting a game of "+GAME+" with "+(string)gNumberOfPlayers+" players.");
        llWhisper(0, "Game play will continue until the first player has scored more than "+(string)gFinalScore+" points, and all other players have had one additional round." );
        llWhisper(0, "The highest scoring player then wins the game.    Enjoy the game!." );
        gCurrentPlayerIdx = -1;
        gTurnScore = 0;
        PassTurn();
    }

    touch_start(integer num)
    {
        integer linkID      = llDetectedLinkNumber(0);
        integer faceID      = llDetectedTouchFace(0);
        key     avatarKey   = llDetectedKey(0);
        string  avatarName  = llDetectedName(0);
        string  ObjectName  = llGetLinkName(linkID);

        if (faceID == TOUCH_INVALID_FACE) {
            llOwnerSay("Sorry, your viewer doesn't support touched faces. You won't be able to play this game");
        } else if ( avatarKey == gCurrentPlayerID ) {
                list    object  = llParseString2List(ObjectName,[":"],[""]);
                string  objType = llList2String(object,0);
                integer objNum  = llList2Integer(object,1);
                if (DEBUG_MODE)  Debug( "PLAY mode:  "+objType+":"+(string)objNum+" touched on face "+(string)faceID );

                if ( objType == "Die" && objNum > 0 ) {
                  integer dieNum = objNum-1;
                  if( GetDieValue(dieNum) < 7 ) {
                      SelectDie(dieNum);
                      gRollScore = CheckPoints(FALSE);
                      if(gOldRollScore != gRollScore ) {
                          UpdateScorePanel(gTurnScore+gRollScore,gRoundScoreLinkID);
                          UpdateScorePanel(gTurnScore+gRollScore+gCurrentPlayerScore,gTotalScoreLinkID);
                          gOldRollScore = gRollScore;
                      }
                      if (DEBUG_MODE)  Debug( "Selected die have "+(string)gRollScore+" points." );
                  }
                } else if ( objType == "Buttons" ) {
                  if ( faceID == 1 ) {  // Roll
                      GamePlay(1);
                  } else if ( faceID == 2 ) {  // Pass
                      if (DEBUG_MODE)  Debug( "PASS clicked" );
                      GamePlay(-1);
                  }
                } else if ( linkID == 1 ) {
                    UpdateLinkIDLists();
                }
        } else {
          llWhisper( 0,"Sorry "+llList2String(llParseString2List(llKey2Name(avatarKey), [" "], []), 0)+", it is not your turn." );
        }
    }


    link_message(integer sender_number, integer number, string message, key id)
    {
        if(DEBUG_MODE) Debug("LinkMessageRX Sender="+(string)sender_number+" |"+message+"| KeyID:"+(string)id);
        list msgParts = llParseString2List(message, [" "], []);
        string cmd = llList2String(msgParts, 0);

        if( cmd == CMD_RESET) {
            UnsitAllAvatars();
            llMessageLinked( LINK_ALL_OTHERS, 0, CMD_RESET, NULL_KEY );
            llResetScript();
        }
    }

    timer()
    {
      llSetTimerEvent(0);
      state IDLE;
    }

    changed(integer change)
    {
      if ( change * CHANGED_LINK ) {
          UpdateLinkIDLists();
      }
    }
}