// CG Facelight controller script v4.1
// By: Cognitive Gears
// Description: Simple script to generate a point light suitable for attachment
// License: Artistic 2.0 - http://www.perlfoundation.org/artistic_license_2_0

// Current version
string gVersionString = "4.1";

vector gLightColor = <1, 1, 1>; // white
float gIntensity   = .5;  // don't want to blind anyone
float gRadius      = 2.0; // Try to keep away shadows on face
float gFalloff     = 0.0; // In testing, best not to allow any falloff


// Link numbers for lights
integer gAllLights = LINK_SET;
integer gHeadlamp  = 1; // The light directly in front of the avatar
integer gLeftlamp  = 2; // The light on the avatar's left
integer gRightlamp = 3; // The light on the avatar's right
integer gSideLamps = 5;

// Message value constants
integer gStateOff       = 0;
integer gStateOn        = 1;
integer gStateSuspended = -1;

// State information
integer gCurrentState    = gStateOn; // Default light to on (until we know better)
integer gTargetLamps     = gHeadlamp; // Used to default to gSideLamps, but this is safer
integer gSuspendOverride = 0; // Controls whether to accept suspends
// Keep a static with the current sub-menu.
// This should be ok since we only have one menu user
string gCurrentSubMenu = "";

// Linked message type definitions
integer gMessageTypeState     = 0;
integer gMessageTypeIntensity = 1;
integer gMessageTypeRadius    = 2;
integer gMessageTypeFalloff   = 3;
integer gMessageTypeColor     = 4;



// Listener
integer gOwnerListenPort       = 1; // Port to listen for owner commands
integer gOwnerListenHandle     = 0; // See llRemoveListener()
integer gOwnerListenMainHandle = 0; // See llRemoveListener()
integer gSuspendListenPort     = -7373; // Port to listen for suspend commands
integer gSuspendListenHandle   = 0; // See llRemoveListener()
integer gMenuListenPort        = -1; // Port to listen for menu clicks (will randomize)
integer gMenuListenHandle      = 0; // See llRemoveListener()

// Other ports
integer gAttachmentListenPort       = -7374; // Port to send attachment commands


// Timer functions
integer gSuspendCheckTimeout = 10; // Season to taste
integer gLastSuspendNoticeTime = 0;


sendLightParams() {
    // Set the params for each prim
    // Note, this function assumes that the root prim is the headlamp

    if( gCurrentState == gStateOn ) {
        // Main light
        if( gTargetLamps == gHeadlamp ) {
            llSetLinkPrimitiveParams( LINK_ALL_CHILDREN, [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOff, gLightColor, gIntensity, gRadius, gFalloff] );
            llSetPrimitiveParams( [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOn, gLightColor, gIntensity * 1.2, gRadius * 1.3, gFalloff] );
        }
        else if( gTargetLamps == gSideLamps ) {
            // Side lights
            llSetPrimitiveParams( [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOff, gLightColor, gIntensity * 1.2, gRadius * 1.3, gFalloff] );
            llSetLinkPrimitiveParams( LINK_ALL_CHILDREN, [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOn, gLightColor, gIntensity, gRadius, gFalloff] );
        }
        else {
            // Assume gAllLights
            // Have to do this in two steps because of the difference for brightness
            llSetLinkPrimitiveParams( LINK_ALL_CHILDREN, [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOn, gLightColor, gIntensity, gRadius, gFalloff] );
            llSetPrimitiveParams( [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOn, gLightColor, gIntensity * 1.2, gRadius * 1.3, gFalloff] );

        }
    }
    else {
        // State isn't on, turn off all lights
        llSetLinkPrimitiveParams( LINK_SET, [PRIM_PHANTOM, TRUE, PRIM_POINT_LIGHT, gStateOff, gLightColor, gIntensity, gRadius, gFalloff] );
    }

    // Send a message to the HUD to update status
    // TODO: Only send a message if the state changed
    if( gCurrentState == gStateOff ) {
        llWhisper(gAttachmentListenPort, "light_off");

    }
    else if( gCurrentState == gStateOn ) {
        llWhisper(gAttachmentListenPort, "light_on");
    }
    else {
        llWhisper(gAttachmentListenPort, "light_suspend");
    }
}

startListeners() {
    // Listen for commands by the owner (or HUD)
    if(gOwnerListenHandle) {
        llListenRemove( gOwnerListenHandle );
    }
    if(gOwnerListenMainHandle) {
        llListenRemove( gOwnerListenMainHandle );
    }
    // Listen on the owners channel as well as channel 0 (like an AO)
    // Note: this is using NULL_KEY because our HUD uses that to communicate as well
    gOwnerListenHandle = llListen( gOwnerListenPort, "", NULL_KEY, "" );
    gOwnerListenMainHandle = llListen( 0, "", llGetOwner(), "" );
    
    // Suspend mode
    if(gSuspendListenHandle) {
        llListenRemove( gSuspendListenHandle );
    }
    gSuspendListenHandle = llListen( gSuspendListenPort, "", NULL_KEY, "" );
}

startMenuListener() {
    // Similar to startListeners, but only runs for menu clicks
    if(gMenuListenHandle) {
        llListenRemove( gMenuListenHandle );
    }

    // Get a random negative channel number, making sure its not 0
    gMenuListenPort   = -1 * ((integer) llFrand(2147483647) + 1 );
    gMenuListenHandle = llListen( gMenuListenPort, "", llGetOwner(), "" );

    // TODO: Timeout listen (but don't conflict with suspender)
}

showMainMenu( key id ) {
    // Displays the main menu to the user
    startMenuListener();
    gCurrentSubMenu = "";
    llDialog(
        id,
        llGetObjectName() + " main configuration\n\nChoose an option, or Ignore to close this window.",
        ["Lights", "Brightness", "Help", "Override", "Color", "Radius"],
        gMenuListenPort );
}

showSubMenu( key id, string command ) {
    // Displays a submenu
    startMenuListener();

    gCurrentSubMenu = command;
    if( command == "Lights" ) {
        llDialog(
            id,
            "Light configuration\n\n\"Front\" is the safest choice for general use.  \"Sides\" provides the best coverage but may look incorrect in situations with more than six light sources (including crowds wearing facelights). \"All\" may be useful in SL photography.\n\nChoose an option, or Ignore to close this window.",
            ["Sides", "All", "<<Back", "Front"],
            gMenuListenPort );
    }
    else if( command == "Brightness" ) {
        llDialog(
            id,
            "Brightness configuration\n\nChoose \"Normal\" for the best coverage, or \"Dim\" for a more natural (but muted) look.  \"Bright\" and \"Blinding\" may be useful for SL photography.\n\nChoose an option, or Ignore to close this window.",
            ["Bright", "Blinding", "<<Back", "Dim", "Normal"],
            gMenuListenPort );
    }
    else if( command == "Override" ) {
        llDialog(
            id,
            "Override configuration\n\nThis switch determins whether to listen to the wishes of property owners for facelight usage. Generally this should be left to \"Off\". Turn to on to ignore any facelight suspenders.\n\nChoose an option, or Ignore to close this window.",
            ["On", "Off", "<<Back"],
            gMenuListenPort );
    }
    else if( command == "Color" ) {
        llDialog(
            id,
            "Color configuration\n\nChoose \"White\" for a full-spectrum light, or choose a distinct color.\n\nChoose an option, or Ignore to close this window.",
            ["Red", "Blue", "<<Back", "Orange", "Pink", "Yellow", "Brown", "Purple", "Green", "White"],
            gMenuListenPort );
    }
    else if( command == "Radius" ) {
        llDialog(
            id,
            "Radius configuration\n\nUse this setting to choose how far the light extends.  \"Normal\" is best in most situations. \"Tiny\" provides coverage when using the Side lights.  \"Big\" and \"Huge\" extend the light further beyond your avatar.\n\nChoose an option, or Ignore to close this window.",
            ["Big", "Huge", "<<Back", "Tiny", "Normal"],
            gMenuListenPort );
    }
    else {
        // Bad command, default to showing the main menu (this also captures "<<Back"
        gCurrentSubMenu = "";
        showMainMenu( id );
    }
}

string getAVParcelHash() {
    // TODO: Better way to determine which parcel we are over
    // To my knowledge, there is nothing like a parcel key, so we need to create a string
    // to compare instead.  This is not necessarily unique for a region, but afaik is as
    // close as we can get.
    list parcelDetails=llGetParcelDetails(llGetPos(),[PARCEL_DETAILS_NAME, PARCEL_DETAILS_OWNER, PARCEL_DETAILS_GROUP, PARCEL_DETAILS_AREA, PARCEL_DETAILS_DESC]);
    return llMD5String( llDumpList2String( parcelDetails, "" ), 0 );
}

startSuspend() {
    // Set the light in suspend mode
    gCurrentState = gStateSuspended;
    // Keep a suspend timer so if we walk out of the area it can come back
    gLastSuspendNoticeTime = llGetUnixTime();
    llSetTimerEvent( gSuspendCheckTimeout );
    sendLightParams();
    llOwnerSay( "Suspending operation based on region policy.");
}

stopSuspend() {
    // Turn off suspend mode
    gCurrentState = gStateOn;
    llSetTimerEvent( 0.0 );
    sendLightParams();
    llOwnerSay( "Resuming operation." );
}


default {
    state_entry() {
        // On state_entry, reset params
        sendLightParams();
        
        // Not likely, but just in case
        llSetStatus( STATUS_CAST_SHADOWS, FALSE );
        // Make all prims invisible
        llSetLinkAlpha( LINK_SET, 0.0, ALL_SIDES );
        
        startListeners();

        llOwnerSay( "Version " + gVersionString + " by Cognitive Gears." );
        
        // Hack for no-script parcels
        llRequestPermissions(llGetOwner(), PERMISSION_TAKE_CONTROLS );
        
        // When we are first setup (or change owner) as which lights to use,
        // as this is an important choice that can affect usability
        llOwnerSay("Starting initial light configuration.  Lights and other settings can be changed through the control panel using the CG Facelight HUD or by saying \"fl menu\" (without the quotes.)");
        showSubMenu( llGetOwner(), "Lights"); 
    }

    run_time_permissions( integer perm ) {
        // Hack for no-script parcels
        llTakeControls( CONTROL_UP, TRUE, TRUE );
    }

    on_rez( integer start ) {
        // On rez (or more importantly attachment, reset params)
        sendLightParams();
        startListeners();
        
        // Hack for no-script parcels
        llRequestPermissions(llGetOwner(), PERMISSION_TAKE_CONTROLS );    
    }

    listen( integer channel, string name, key id, string message ) {
        integer menuChannel = 0;
        if( channel == gMenuListenPort && message != "" ) {
            if( message == "Lights" || message == "Brightness" || message == "Override" || message == "Color" || message == "Radius" || message == "<<Back" ) {
                showSubMenu( id, message );
            }
            else {
                // Change the message to contain the facelight prefix and continue processing
                // Here be dragons
                string prefix = llToLower( gCurrentSubMenu );
                if( prefix == "lights" ) {
                    // we use a shortcut for lights, no need to set a prefix
                    prefix = "";
                } else if( prefix != "" ) {
                    // Add a space
                    prefix = prefix + " ";
                }
                message = "fl " + prefix + llToLower( message );
                menuChannel = 1;
            }
        }
        
        if( channel == gOwnerListenPort || channel == 0 || menuChannel ) {
            if( id != llGetOwner() )
            {
                id = llGetOwnerKey( id );
            }
            if( id != llGetOwner() ) {
                // We only listen to our owner or objects owned by our owner
                return;
            }
            list messageList = llParseString2List( message, [" "], [ ] );
            if(llGetListLength( messageList ) != 2 && llGetListLength( messageList) != 3) {
                // We are expecting a list of length 2 or 3 with an arg
                return;
            }

            string dObjStr = llToLower( llList2String( messageList, 0 ) );
            if( dObjStr != "facelight" && dObjStr != "facelamp" && dObjStr != "fl" ) {
                // All commands start with facelight (we'll also accept facelamp)
                return;
            }

            string argStr = llToLower( llList2String( messageList, 1 ) );
            if( argStr == "on" && gCurrentState != gStateSuspended ) {
                llOwnerSay("Turning on.");
                gCurrentState = gStateOn;
            }
            else if( argStr == "off" && gCurrentState != gStateSuspended ) {
                llOwnerSay("Turning off.");
                gCurrentState = gStateOff;
            }
            else if( argStr == "force" ) {
                llOwnerSay("Forcing on.");
                // Force it back on
                gCurrentState = gStateOn;
            }
            else if( argStr == "all" ) {
                llOwnerSay("Setting to all lights.");
                // Turn on all lights
                gTargetLamps = gAllLights;
            }
            else if( argStr == "sides" || argStr == "side" ) {
                llOwnerSay("Setting to sides only.");
                // Turn on side lamps
                gTargetLamps = gSideLamps;
            }
            else if( argStr == "front" ) {
                llOwnerSay("Setting to front only.");
                // Turn on headlamp
                gTargetLamps = gHeadlamp;
            }
            else if( argStr == "menu" ) {
                llOwnerSay("Displaying the menu.");
                showMainMenu( id );
            }
            else if( argStr == "help" ) {
                llGiveInventory(id, "CG Facelight User Guide");
            }
            else if( argStr == "override" && llGetListLength( messageList) == 3 ) {
                string overrideState = llToLower( llList2String( messageList, 2 ) );
                if( overrideState == "on" ) {
                    gSuspendOverride = 1;
                    if( gCurrentState == gStateSuspended ) {
                        // Turn off suspend mode
                        stopSuspend();
                    }
                    llOwnerSay("Turning on suspend override mode.");
                }
                else if( overrideState == "off" ) {
                    gSuspendOverride = 0;
                    llOwnerSay("Turning off suspend override mode.");

                }
                else {
                    llOwnerSay("Unknown value, \"" + overrideState + "\"");
                }
            }
            else if( argStr == "brightness" && llGetListLength( messageList) == 3 ) {
                string brightness = llToLower( llList2String( messageList, 2 ) );
                if( brightness == "dim" ) {
                    gIntensity = .2;
                }
                else if( brightness == "normal" ) {
                    gIntensity = .5;
                }
                else if( brightness == "bright" ) {
                    gIntensity = .7;
                }
                else if( brightness == "blinding" ) {
                    gIntensity = 10.0;
                }

                else {
                    llOwnerSay("Unknown value, \"" + brightness + "\"");
                }
            }
            else if( argStr == "radius" && llGetListLength( messageList) == 3 ) {
                string radius = llToLower( llList2String( messageList, 2 ) );
                if( radius == "tiny" ) {
                    gRadius = 1;
                }
                else if( radius == "normal" ) {
                    gRadius = 2;
                }
                else if( radius == "big" ) {
                    gRadius = 4;
                }
                else if( radius == "huge" ) {
                    gRadius = 10;
                }
                else {
                    llOwnerSay("Unknown value, \"" + radius + "\"");
                }
            }
            else if( argStr == "color" && llGetListLength( messageList) == 3 ) {
                // TODO: Needs refactoring
                string color = llToLower( llList2String( messageList, 2 ) );
                if( color == "white" ) { gLightColor = <1, 1, 1>; }
                else if( color == "red" ) { gLightColor = <1.0,0.0,0.0>; }
                else if( color == "blue" ) { gLightColor = <0.0, 0.0, 1.0>; }
                else if( color == "orange" ) { gLightColor = <0.8, 0.4, 0.0>; }
                else if( color == "pink" ) { gLightColor = <1.0, 0.6, 1.0>; }
                else if( color == "yellow" ) { gLightColor = <1.0, 1.0, 0.0>; }
                else if( color == "brown" ) { gLightColor = <0.4, 0.2, 0.0>; }
                else if( color == "green" ) { gLightColor = <0.0, 1.0, 0.0>; }
                else if( color == "purple" ) { gLightColor = <0.6, 0.0, 0.8>; }
                else {
                    llOwnerSay("Unknown value, \"" + color + "\"");
                }
            }
            else {
                llOwnerSay("Unknown command, \"" + argStr + "\"");
            }
        }
        else if( channel == gSuspendListenPort ) {

            // Check owner id to validate the suspender has rights to suspend lights
            // Either the suspender needs to be deeded to the group and the land group
            // owned, or the suspender must belong to the same owner.
            list parcelDetails = llGetParcelDetails( llGetPos(), [PARCEL_DETAILS_OWNER] );
            key parcelOwner = llList2Key( parcelDetails, 0 ); // Could be an id or group
            if( id != parcelOwner ) {
                id = llGetOwnerKey( id );
                if( id != parcelOwner ) {
                    // Oops, the owner/group doesn't match.  Bail
                    return;
                }
            }
            
            
            if( gSuspendOverride != 0 ) {
                // Override is on, so don't suspend
                return;
            }
            else if( gCurrentState == gStateSuspended ) {
                // Just update the time
                gLastSuspendNoticeTime = llGetUnixTime();
                return;
            }
            // Suspender
            list messageList = llParseString2List( message, [" "], [ ] );
            string dObjStr = llToLower( llList2String( messageList, 0 ) );
            if( dObjStr != "suspend_light" && dObjStr != "suspend_lights" ) {
                return; // We don't handle any other options (for now)
            }
            // TODO: resume_light?  We don't want blinkie lights
            if( llGetListLength( messageList ) == 2 ) {
                // Second argument should be a parcel hash
                if( llList2String( messageList, 1 ) != getAVParcelHash() ) {
                    return; // This message is not for our parcel
                }
            }
            startSuspend();
            return; // We already sent light params from startSuspend
        }
        sendLightParams();    
    }

    timer() {
        if( llGetUnixTime( ) >= gLastSuspendNoticeTime + gSuspendCheckTimeout ) {
            // We haven't heard from the suspender in a while, turn off suspend mode
            stopSuspend();
        }
    }

    changed( integer changed_num ) {
        if( changed_num & CHANGED_OWNER ) {
            // We have a new owner, reset everything
            llResetScript();
        }
        
        // handle teleport (remove suspend mode)
        if( ( changed_num & CHANGED_TELEPORT || changed_num & CHANGED_REGION ) && gCurrentState == gStateSuspended ) {
            stopSuspend();
        }
    }
} 